"""Component utility functions."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
import json
import os
from pathlib import Path
from typing import Optional
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display.display_options_io import read_display_options_from_json, write_display_options_to_json
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.settings import SettingsManager

# 4. Local modules


def duplicate_display_opts(new_directory: Path | str, old_display_options: str | Path) -> dict:
    """
    Duplicate display options.

    Args:
        new_directory: Path to the directory where the new display options will be saved. Will typically be a component
            directory.
        old_display_options: Path to the .json file containing the display options to duplicate.

    Returns:
        Dictionary containing the duplicated display options.
    """
    fname = os.path.join(new_directory, old_display_options)
    json_dict = read_display_options_from_json(fname)
    if 'uuid' in json_dict:
        json_dict['uuid'] = str(uuid.uuid4())
        json_dict['comp_uuid'] = os.path.basename(new_directory)
        categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
        categories.from_dict(json_dict)
        write_display_options_to_json(fname, categories)
    return json_dict


def read_default_display_options_from_registry(registry_key: str) -> Optional[CategoryDisplayOptionList]:
    """
    Retrieve the default display options from the registry.

    Args:
        registry_key: The registry key of the display options to load.

    Returns:
        The default options from the registry, or None if not saved yet.
    """
    settings_manager = SettingsManager()
    json_text = settings_manager.get_setting('xmsgmi', registry_key)
    if not json_text:
        return None

    json_dict = json.loads(json_text)
    categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
    categories.from_dict(json_dict)
    return categories


def new_component_dir() -> Path:
    """
    Get a new component directory.

    The returned directory will already exist.
    """
    comp_uuid = str(uuid.uuid4())
    comp_dir = os.path.join(XmEnv.xms_environ_temp_directory(), 'Components', comp_uuid)
    os.makedirs(comp_dir)
    return Path(comp_dir)


def invert_comp_to_xms(comp_to_xms: dict[int, list[int]]) -> dict[int, int]:
    """
    Convert a mapping of component_id -> [feature_id] to a mapping of feature_id -> component_id.

    This assumes that each feature_id is associated with only one component_id. If it associated with more than one in
    the input, the result will associate it with one arbitrarily.

    Args:
        comp_to_xms: Mapping from component ID to list of feature IDs.

    Returns:
        Mapping from feature ID to component ID.
    """
    xms_to_comp = {}
    for comp_id in comp_to_xms:
        for feature_id in comp_to_xms[comp_id]:
            xms_to_comp[feature_id] = comp_id
    return xms_to_comp
