"""The model control properties dialog."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
import pandas as pd
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules
from xms.guipy.dialogs.help_getter import HelpGetter
from xms.guipy.dialogs.table_dialog import TableDialog
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor
from xms.tool_core.table_definition import TableDefinition

# 4. Local modules


class BaseDialog(XmsDlg):
    """A dialog for editing model control properties."""
    def __init__(self, parent, dlg_name):
        """Initializes the dialog, sets up the ui.

        Args:
            parent (QWidget): Parent window
            dlg_name (str): string used in the registry
        """
        super().__init__(parent, dlg_name)
        default = 'https://xmswiki.com/wiki/SMS:GenericModelDialog'
        self.help_getter = HelpGetter(key=self._dlg_name, default=default, dialog_help_url='')

    def on_btn_click_xy(self, xy_id: int, axis_titles: list[str], use_dates: bool, stair_step: bool) -> int:
        """
        Display a curve editor to view/edit a curve.

        Args:
            xy_id: ID of the curve to display.
            axis_titles: Curve axis titles.
            use_dates: Whether to use dates.

        Returns:
            ID of the new curve if edited, or `xy_id` if canceled.

        :meta private:  This is only public because it's a slot. User code shouldn't use it.
        """
        x, y = self._get_curve(xy_id, use_dates)
        pd_dict = {
            axis_titles[0]: x,
            axis_titles[1]: y,
        }
        df = pd.DataFrame(pd_dict)
        df.index += 1

        dlg = XySeriesEditor(df, '', parent=self, stair_step=stair_step)
        if not dlg.exec():
            return xy_id

        df = dlg.model.data_frame
        if len(df) < 1:
            return -1

        curve_id = self._add_curve(df[axis_titles[0]].to_list(), df[axis_titles[1]].to_list(), use_dates)
        return curve_id

    def on_btn_click_table(self, label: str, table_definition: TableDefinition, value):
        """
        Display a table editor to view/edit a table.

        Args:
            label: Parameter label, used for the table dialog title.
            table_definition: Defines the table column types.
            value: 2D list of values.

        Returns:
            The edited table values, as a 2D list.

        :meta private:  This is only public because it's a slot. User code shouldn't use it.
        """
        df = table_definition.to_pandas(rows=value)
        dialog = TableDialog(label, table_definition, df, parent=self, dlg_name=self._package_name())
        if dialog.exec() == QDialog.Rejected:
            return None
        df = dialog.get_values()
        for column in df:
            if df[column].dtype == np.dtype('datetime64[ns]'):
                df[column] = df[column].astype(dtype=str)
        return df.values.tolist()  # Convert dataframe to list of lists, row-wise

    def accept(self):
        """
        Handle the user clicking OK.

        :meta private:  This is only public because it's a slot. User code shouldn't use it.
        """
        super().accept()
