"""BcCoverageComponent class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from functools import cached_property

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.components.coverage_component import CoverageComponent
from xms.gmi.data.generic_model import Section
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.gssha.data import bc_generic_model
from xms.gssha.data.bc_coverage_data import BcCoverageData


class BcCoverageComponent(CoverageComponent):
    """A component for boundary conditions."""
    def __init__(self, main_file: str) -> None:
        """Initializes the class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)

        self._inactive_group_name = ''  # Overriden from base class cause our group names are strings
        self._multiple_active_group_name = '$MGROUP$'  # Overridden from base class cause our group names are strings
        message = (
            'This GSSHA interface is in "beta". Feedback is appreciated. Please report any bugs to'
            ' <a href=\'mailto:support@aquaveo.com\'>support@aquaveo.com</a>.'
        )
        style_sheet = 'QLabel { background-color : rgb(200, 200, 150); color : rgb(0, 0, 0); }'
        self._banner = (message, style_sheet)
        self.polygon_commands = []  # We don't have any polygon properties

    @cached_property
    def data(self) -> BcCoverageData:
        """Returns the component's data manager.

        Returns:
            BcCoverageData: The data manager for this component.
        """
        return BcCoverageData(self.main_file)

    def _section(self, target_type: TargetType) -> Section:
        """
        Get a GMI Section from the coverage's generic model based on its `TargetType`.

        Args:
            target_type: `TargetType` for the desired section.

        Returns:
            Section: The section for the feature type.
        """
        return bc_generic_model.create().section_from_target_type(target_type)
