"""Handles creating the generic model for the boundary conditions."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel

# 4. Local modules
from xms.gssha.data.generic_model_base import GenericModelCreatorBase

# Constants
# Dict of overland flow type strings -> names of parameters with the data (float if constant, xy series if variable)
olf_type_params = {
    'Constant slope': 'constant_slope',
    'Constant stage (water surface elevation)': 'constant_stage',
    'Variable stage (water surface elevation)': 'variable_stage',
    'Variable flow (cms discharge)': 'variable_flow_cms',
    'Variable flow (cfs discharge)': 'variable_flow_cfs',
}
FIRST_VARIABLE_OLF_TYPE = 2  # Index of the first variable overland flow type


class ChannelType:
    """Types of channels."""
    TRAPEZOIDAL = 'Trapezoidal'
    CROSS_SECTION = 'Cross-section'


def create() -> GenericModel:
    """Creates and returns the generic model.

    Returns:
        (GenericModel): The generic model.
    """
    creator = BcGenericModelCreator()
    return creator.create()


class BcGenericModelCreator(GenericModelCreatorBase):
    """Creates the GenericModel for the BC coverage.

    By convention, parameter names are the same as the GSSHA card and are upper case. If there is no matching GSSHA
    card, the parameter names are lower case.
    """
    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self._gm = GenericModel(exclusive_point_conditions=True, exclusive_arc_conditions=True)
        self._descriptions: dict[str, str] = _descriptions()

    def create(self) -> GenericModel:
        """Creates and returns the generic model.

        Returns:
            (GenericModel): The generic model.
        """
        self._add_point_parameters()
        self._add_arc_parameters()
        return self._gm

    def _add_point_parameters(self) -> None:
        """Adds the point parameters."""
        self._section = self._gm.point_parameters
        self._add_overland_flow_group()

    def _add_overland_flow_group(self) -> None:
        """Adds the overland flow parameters."""
        self._add_group('overland_flow', 'Overland Flow BC')
        olf_type_strs = get_olf_types('all')
        flags = {option: False for option in olf_type_strs}
        o = self._option('overland_flow_type', 'Overland Flow BC Type', olf_type_strs[0], olf_type_strs)

        p = self._float('constant_slope', olf_type_strs[0], 0.0)
        p.add_dependency(o, self._flags(flags, olf_type_strs[0]))
        p = self._float('constant_stage', olf_type_strs[1], 0.0)
        p.add_dependency(o, self._flags(flags, olf_type_strs[1]))
        p = self._xy_series('variable_stage', olf_type_strs[2], ['Time (min)', 'Stage elevation (m)'])
        p.add_dependency(o, self._flags(flags, olf_type_strs[2]))
        p = self._xy_series('variable_flow_cms', olf_type_strs[3], ['Time (min)', 'Discharge (cms)'])
        p.add_dependency(o, self._flags(flags, olf_type_strs[3]))
        p = self._xy_series('variable_flow_cfs', olf_type_strs[4], ['Time (min)', 'Discharge (cfs)'])
        p.add_dependency(o, self._flags(flags, olf_type_strs[4]))

    def _add_arc_parameters(self) -> None:
        """Adds the arc parameters."""
        self._section = self._gm.arc_parameters
        self._add_channel_group()
        self._add_overland_flow_group()

    def _add_channel_group(self) -> None:
        """Adds the channel group."""
        self._add_group('channel', 'Channel')
        opts = [ChannelType.TRAPEZOIDAL, ChannelType.CROSS_SECTION]
        o = self._option('channel_type', 'Channel type', opts[0], opts)
        flags = {ChannelType.TRAPEZOIDAL: False, ChannelType.CROSS_SECTION: False}
        self._float('mannings_n', 'Manning\'s n (MANNINGS_N)', 0.0)

        # Trapezoidal
        flags_trapezoidal = self._flags(flags, opts[0])
        p = self._float('bankfull_depth', 'Depth (m) (BANKFULL_DEPTH)', 0.0)
        p.add_dependency(o, flags_trapezoidal)
        p = self._float('bottom_width', 'Bottom width (m) (BOTTOM_WIDTH)', 0.0)
        p.add_dependency(o, flags_trapezoidal)
        p = self._float('side_slope', 'Side slope (H:V) (SIDE_SLOPE)', 0.0)
        p.add_dependency(o, flags_trapezoidal)

        # Cross section
        flags_cross_section = self._flags(flags, opts[1])
        p = self._xy_series('cross_section', 'Cross section', ['X', 'Y'])
        p.add_dependency(o, flags_cross_section)

        # Stream network outlet
        self._bool('most_downstream_arc', 'Most downstream arc of stream network', False)
        # Output hydrograph
        self._bool('hydrograph_down', 'Output hydrograph at downstream end', False)
        # Input hydrograph
        a = self._bool('specify_point_source', 'Specify point source at upstream end', False)
        p = self._xy_series('point_source_xy', 'Hydrograph', ['Time (min)', 'Flow (cms)'])
        p.add_dependency(a, {True: True, False: False})


def _descriptions() -> dict[str, str]:
    """Returns a dict of descriptions."""
    # yapf: disable
    return {
        # BC Coverage

        # Points
        # Overland flow bc
        'overland_flow': '',
        'overland_flow_type': '',
        'constant_slope': '',
        'constant_stage': '',
        'variable_stage': '',
        'variable_flow_cms': '',
        'variable_flow_cfs': '',
        'overland_depth_output': '',
        'overland_wse_output': '',
        'model_linkage_output': '',

        # Arcs
        # trapezoidal
        'channel': '',
        'channel_type': '',
        'mannings_n': '',
        'bankfull_depth': '',
        'bottom_width': '',
        'side_slope': '',
        # cross-section
        'max_conveyance_depth': '',
        'cross_section': '',
        'define_interpolation_points': '',
        # most downstream arc
        'most_downstream_arc': 'The arc that is at the bottom (most downstream) of the stream network. There must be '
                               'one and only one arc with this property.',
        # outlet hydrograph
        'hydrograph_down': 'Output the hydrograph at the downstream end of the arc',
        # point source
        'specify_point_source': 'Specify the input hydrograph at the upstream end of the arc',
        'point_source_xy': 'The input hydrograph at the upstream end of the arc',
    }
    # yapf: enable


def get_olf_types(which: str) -> list[str]:
    """Returns the list of overland flow type options.

    Args:
        which: 'all' returns all types, 'variable' returns variable types.

    Returns:
        See description
    """
    olf_type_strs = [key for key, _ in olf_type_params.items()]
    match which:
        case 'all':
            return olf_type_strs
        case 'variable':
            return olf_type_strs[FIRST_VARIABLE_OLF_TYPE:]
        case _:
            raise ValueError('get_olf_types() - "which" must be one of: "all", "variable"')
