"""Block class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Group, Type

# 4. Local modules


class Block:
    """Helps us write to text files."""
    def __init__(self, file, log, heading: str, align: 'int | None' = None):
        r"""Initializer.

        Args:
            file: The file to write to.
            log: The logger.
            heading (str): The block heading. If blank, no heading is written. If '\n', only a blank line is written.
            align: Column to align values at. If -1, no alignment is made.
        """
        self._file = file
        self._log = log
        self._heading: str = heading
        self._lines: list[str] = []
        self._align: 'int | None' = align

    def __enter__(self):
        """Called when we enter, and writes a message to the log."""
        if self._heading and self._heading != '\n' and self._log:
            self._log.info(f'Writing {self._heading}')
        return self

    def __exit__(self, _type, value, traceback) -> None:
        """Called when we exit, and dumps whatever we cached to the file.

        Args:
            _type: If there was an exception, the exception type.
            value: If there was an exception, the error string.
            traceback: If there was an exception, the traceback object.
        """
        if not self._lines:
            return

        if self._heading:
            if self._heading == '\n':
                self._lines.insert(0, '')
            else:
                self._lines.insert(0, f'# {self._heading}')
                self._lines.insert(0, '')

        if self._file:  # (can be None when testing)
            self._file.writelines(line + '\n' for line in self._lines)

    def write(self, group: 'Group | None', name: str, value=None, quotes: bool = False) -> None:
        """Helper function to write a line to the file.

        A variety of combination of arguments can be passed:

        1) group, name, and value: (e.g. for booleans to print the value passed but only if parameter value is true)
        2) group, name: prints the name and parameter value. For booleans, only prints the name.
        3) name, value: prints the name and the value
        4) name: prints the name
        5) value: prints the value

        Args:
            group (Group|None): The generic model group. If not None, value is obtained from this group.
            name (str): The generic model parameter name, which is also the GSSHA CARD.
            value: Optional. If given, the value to write after the card, else value obtained from group.
            quotes: If True, value is written in double quotes.
        """
        # Get the name and the value
        _name = None
        _value = None
        if group and name and value is not None:
            # Use the name and the value passed in, but only if the parameter value is truthy
            if name in group.parameter_names and group.parameter(name).value:
                _name = name
                _value = value
        elif group and name:
            if name in group.parameter_names:
                if group.parameter(name).parameter_type != Type.BOOLEAN:
                    _name = name
                    _value = group.parameter(name).value
                elif group.parameter(name).value:  # For booleans, we just write the name (if it's true)
                    _name = name
        elif name and value:
            _name = name
            _value = value
        elif name:
            _name = name
        elif value:
            _value = value
        else:
            raise ValueError('Invalid combination of arguments.')

        # Write
        value = f'"{_value}"' if _value and quotes else _value
        if _name is not None and value is not None:
            if self._align is not None:
                # align - 1 because we hard code guarantee one space between name and value
                self._lines.append(f'{_name.upper():<{self._align - 1}} {value}')
            else:
                self._lines.append(f'{_name.upper()} {value}')
        elif _name is not None:
            self._lines.append(f'{_name.upper()}')
        elif value is not None:
            self._lines.append(f'{value}')
