r"""Reads the `STREAM_CELL <ci>`_ (.gst) file, which tells GSSHA how the streams link to the grid.

.. _ci: https://www.gsshawiki.com/Surface_Water_Routing:Channel_Routing

::

    GRIDSTREAMFILE
    STREAMCELLS 108
    CELLIJ     1      31
    NUMNODES   2
    LINKNODE   5      77     1.0
    LINKNODE   5      76     0.3020346471717026
    CELLIJ     2      31
    NUMNODES   5
    LINKNODE   5      76     0.6979653528282974
    LINKNODE   5      75     1.0
    LINKNODE   5      74     1.0
    LINKNODE   5      73     1.0
    LINKNODE   5      72     0.7556114331203649
    .
    .
    .
"""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import logging
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.gssha.file_io import io_util


def read(file_path: Path | str) -> dict:
    """Reads the STREAM_CELL (.gst) file and returns the info as a dict.

    Args:
        file_path: .gst file path.
    """
    reader = StreamCellFileReader(file_path)
    return reader.read()


class StreamCellFileReader:
    """Reads the STREAM_CELL (.gst) file."""
    def __init__(self, file_path: Path | str) -> None:
        """Initializes the class.

        Args:
            file_path (str | Path): .gst file path.
        """
        super().__init__()
        self._file_path: Path = Path(file_path)
        self._log = logging.getLogger('xms.gssha')

    def read(self) -> dict:
        """Reads the STREAM_CELL (.gst) file and returns a dict of arcs and the cells they intersect."""
        self._log.info('Reading .gst file...')
        arc_cells: dict[int, set[tuple[int, int]]] = {}
        with open(self._file_path, 'r') as file:
            for line in file:
                card, card_value = io_util.get_card_and_value(line)
                if card == 'CELLIJ':
                    words = card_value.split()
                    i_j = int(words[0]), int(words[1])
                elif card == 'LINKNODE':
                    link_number = int(card_value.split(maxsplit=1)[0])
                    arc_cells.setdefault(link_number, set()).add(i_j)
        return arc_cells
