"""GUI utility functions."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import os
from pathlib import Path
import subprocess

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import TreeNode
from xms.guipy.dialogs import message_box

# 4. Local modules


def open_file_in_default_app_or_notepad(file_path: Path, parent_window) -> None:
    """Opens the file in the default app or, if none, Notepad.

    Args:
        file_path (Path): The file path.
        parent_window: The parent window.
    """
    try:
        os.startfile(file_path, 'open')
    except OSError:
        try:
            subprocess.Popen(['notepad', str(file_path)])
        except OSError:
            suffix = file_path.suffix
            message = f'Could not open file "{str(file_path)}". Check for app associated with "{suffix}" files.'
            message_box.message_with_ok(parent=parent_window, message=message, app_name='GMS')


def find_unique_child_name(parent: TreeNode, starting_name: str) -> str:
    """Returns a name, built from starting_name, that is unique among parent's children.

    Appends ' (2)', ' (3)'...  until name is unique. See get_unique_name().

    Args:
        parent: Parent tree node.
        starting_name: The suggested, or starting name.

    Returns:
        The unique name (which may be starting_name).
    """
    starting_name = starting_name.strip()  # Remove leading and trailing whitespace
    names = {child.name for child in parent.children}  # Get set of child names
    return find_unique_name(names, starting_name)


def find_unique_name(names: set[str], starting_name: str) -> str:
    """Returns a name, built from starting_name, that is unique among the set of names.

    Appends ' (2)', ' (3)'...  until name is unique. Adapted from gnGetUniqueName() from xms.

    Args:
        names: Set of names to compare against.
        starting_name: The suggested, or starting name.

    Returns:
        The unique name (which may be starting_name).
    """
    if starting_name not in names:
        return starting_name

    name = starting_name
    while name in names:
        open_parens = name.rfind('(')  # Use rfind to find the last '('
        close_parens = name.rfind(')', open_parens)  # Use rfind to find the last ')'
        if open_parens >= 0 and close_parens >= 0 and open_parens < close_parens:
            try:
                num = int(name[open_parens + 1:close_parens]) + 1  # Will throw if thing isn't an int
                name = name[0:open_parens + 1] + str(num) + ')'
            except ValueError:
                name = name + ' (2)'
        else:
            name = name + ' (2)'
    return name
