"""SimRunner class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, ExecutableCommand, Query
from xms.components.bases.run_base import RunBase
from xms.data_objects.parameters import Simulation
from xms.guipy.dialogs import message_box

# 4. Local modules
from xms.gssha.file_io import solution_reader
from xms.gssha.misc.type_aliases import ActionRv


class SimRunner(RunBase):
    """Class that handles running GSSHA."""
    def __init__(self, dummy_mainfile: str = '') -> None:
        """Initializes the class.

        Args:
            dummy_mainfile (str): Unused. Just to keep constructor consistent with component classes.
        """
        super().__init__()

    def get_executables(self, sim: Simulation, query: Query, file_location: str) -> list[ExecutableCommand]:
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim: The Simulation you want to load the solution for.
            query: a Query object to communicate with XMS.
            file_location: The location of input files for the simulation.

        Returns:
            The executable objects to run and the action requests that go with it.
        """
        gssha_path = Path(file_location) / f'{sim.name}.gssha'
        if not gssha_path.is_file():
            message_box.message_with_ok(
                parent=None,
                message=f'No simulation found at "{gssha_path}". Have you saved the simulation?',
                app_name='SMS'
            )
            return []

        executable_str = 'GSSHA'
        cmd = ExecutableCommand(
            executable=executable_str,  # This is what appears in the XMS model executable preferences
            model='GSSHA',
            executable_order=0,
            display_name=executable_str,
            run_weight=100,
            # progress_script='xml_entry_points/sim_progress.py'
        )
        cmd.add_commandline_arg(f'{str(gssha_path.name)}')
        cmd.add_solution_file(self.get_solution_load_actions(sim, query, file_location)[0])
        return [cmd]

    def get_solution_load_actions(self, sim: Simulation, query: Query, file_location: str) -> list[ActionRequest]:
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim: The Simulation you want to load the solution for.
            query: a Query object to communicate with XMS.
            file_location: The location of input files for the simulation.

        Returns:
            The executable objects to run and the action requests that go with it.
        """
        gssha_path = Path(file_location) / f'{sim.name}.gssha'
        if not gssha_path.is_file():
            message_box.message_with_ok(
                parent=None, message=f'Could not find the simulation at "{gssha_path}".', app_name='SMS'
            )
            return []

        load_sol = ActionRequest(
            modality='MODAL',
            class_name=self.__class__.__name__,
            module_name=self.__module__,
            method_name='read_solution',
            parameters={'gssha_path': str(gssha_path)}
        )
        return [load_sol]

    def read_solution(self, query: Query, params: dict, win_cont: QWidget) -> ActionRv:
        """Reads the solution.

        Args:
            query: a Query object to communicate with GMS.
            params: Generic map of parameters. Contains the structures for various components that are required for
                adding vertices to the Query Context with Add().
            win_cont: The window container.

        Returns:
            ActionRv
        """
        gssha_path = Path(params[0]['gssha_path'])
        rv = solution_reader.read(gssha_path, query, win_cont)
        return rv
