"""Runs cleandam."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from dataclasses import dataclass
from logging import Logger
from pathlib import Path
import subprocess

# 2. Third party modules
from geopandas import GeoDataFrame
import numpy as np

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.constraint import Grid
from xms.core.filesystem import filesystem
from xms.executables.gssha import get_executable_paths
from xms.grid.ugrid import UGrid

# 4. Local modules
from xms.gssha.file_io import io_util


@dataclass
class ArgVals:
    """Simple struct to collect the argument values."""
    coverage: GeoDataFrame | None = None
    clean_flat_areas: bool = True
    clean_digital_dams: bool = True
    fill_leftover_problems: bool = True
    new_dataset_name: str = ''
    co_grid: Grid | None = None
    ugrid: UGrid | None = None


def run(values, arg_vals: ArgVals, query: Query, logger: Logger) -> Path:
    """Writes the files needed by cleandam.exe, runs it, and returns the path of the output file."""
    temp_dir = _make_temp_dir()

    abc = _build_options_string(arg_vals)

    # write elevation file
    elevfile = temp_dir / 'elevfile.asc'
    io_util.write_grass_file(arg_vals.co_grid, arg_vals.ugrid, values, elevfile, ints=False)

    # write watershed mask file
    watmask = temp_dir / 'watmask.asc'
    io_util.write_mask_file(arg_vals.co_grid, arg_vals.ugrid, watmask)

    # Define output file path
    outfile = temp_dir / 'outfile.asc'

    # write depression mask file
    depmask = temp_dir / 'depmask.asc'
    stream_mask = np.zeros(arg_vals.ugrid.cell_count, dtype=int)
    if arg_vals.coverage is not None:
        stream_mask = io_util.get_stream_mask(query, arg_vals.co_grid, arg_vals.ugrid, arg_vals.coverage)
    io_util.write_grass_file(arg_vals.co_grid, arg_vals.ugrid, stream_mask, depmask, ints=True)

    # Get command line and current working directory (cwd)
    cleandam_exe_path = Path(_get_exe_file_path('cleandam.exe'))
    cleandam_exe_path = f'{str(cleandam_exe_path.as_posix())}'  # forward slashes, quoted
    args = [cleandam_exe_path, abc, elevfile.name, watmask.name, outfile.name, depmask.name]
    cwd = f'{str(temp_dir.as_posix())}'  # forward slashes, quoted
    logger.info(f'Current working directory: {cwd}')
    logger.info(f'Command line: {" ".join(args)}')

    # Launch cleandam.exe
    rv = subprocess.Popen(args, stdout=subprocess.PIPE, stdin=subprocess.PIPE, cwd=cwd)

    # Echo output to tool progress
    for line in rv.stdout:
        try:  # cleandam.exe spits out some garbage, so we need this
            line = line.decode('utf-8').rstrip()
        except UnicodeDecodeError:
            pass
        logger.info(line)

    return outfile


def _build_options_string(arg_vals: ArgVals) -> str:
    """Builds and returns the 'abc' options string."""
    abc = ''
    if arg_vals.clean_flat_areas:
        abc += 'a'
    if arg_vals.clean_digital_dams:
        abc += 'b'
    if arg_vals.fill_leftover_problems:
        abc += 'c'
    return abc


def _make_temp_dir() -> Path:
    """Makes a temporary directory."""
    temp_dir = Path(filesystem.temp_filename())
    filesystem.make_or_clear_dir(temp_dir)
    return temp_dir


def _get_exe_file_path(exe_file_name: str) -> Path | None:
    """Finds and returns the path to cleandam.exe."""
    exe_paths = get_executable_paths()
    exe_path = None
    for path in exe_paths:
        if path.lower().endswith(exe_file_name):
            exe_path = path
            break
    return exe_path
