"""MappingTableTool class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules
from xms.api.tree import TreeNode
from xms.constraint import Grid
from xms.gdal.rasters import RasterInput
from xms.gdal.vectors import VectorInput
from xms.grid.ugrid import UGrid
from xms.tool_core import Argument, IoDirection, Tool
from xms.tool_core.table_definition import ChoicesColumnType, InputFileColumnType, StringColumnType, TableDefinition

# 4. Local modules
from xms.gssha.tools import tool_util
from xms.gssha.tools.algorithms import mapping_table_creator as mc

# Constants
ARG_INPUT_USE_SOIL_DATA = 0
ARG_INPUT_SOIL_TABLE = 1
ARG_INPUT_USE_LAND_USE_DATA = 2
ARG_INPUT_LAND_USE_TABLE = 3
ARG_INPUT_INDEX_MAP_NAME = 4
ARG_OUTPUT_INDEX_MAP_DATASET = 5

CHOICES_COL = 4


class MappingTableTool(Tool):
    r"""Creates an index map and mapping table from the UGrid, soil data, and land use data."""
    def __init__(self) -> None:
        """Initializes the class."""
        super().__init__(name='Mapping Table')
        self.mapping_table_name = ''
        self.ugrid_node: 'TreeNode | None' = None  # So we can get a unique tree name for the dataset
        self.co_grid: 'Grid | None' = None
        self.ugrid: 'UGrid | None' = None
        self.run_results: mc.RunResults = mc.RunResults()

        self._lu_shp_fields: dict[str, list[str]] = {}  # Landuse shapefile fields

        # For testing
        # import os
        # os.environ['XMSTOOL_GUI_TESTING'] = 'YES'

    def initial_arguments(self) -> list[Argument]:
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        soil_table_def, soil_df = _create_soils_table()
        lu_table_def, lu_df = _create_land_use_table()
        arguments = [
            self.bool_argument('use_soil_data', 'Use soil data', value=False),
            self.table_argument('soil_table', 'Soil shapefiles', value=soil_df, table_definition=soil_table_def),
            self.bool_argument('use_land_use_data', 'Use land use data', value=False),
            self.table_argument('land_use_table', 'Land use files', value=lu_df, table_definition=lu_table_def),
            self.string_argument('index_map_name', 'Index map (dataset) name', value='index-map'),
            self.dataset_argument('output_dataset', hide=True, optional=True, io_direction=IoDirection.OUTPUT),
        ]
        return arguments

    def enable_arguments(self, arguments: list[Argument]) -> None:
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            arguments: The tool arguments.
        """
        # Hide soil shapefile tables if not using soil data
        use_soil_data = arguments[ARG_INPUT_USE_SOIL_DATA].value
        arguments[ARG_INPUT_SOIL_TABLE].hide = not use_soil_data

        # Hide land use data if not using it
        use_land_use_data = arguments[ARG_INPUT_USE_LAND_USE_DATA].value
        arguments[ARG_INPUT_LAND_USE_TABLE].hide = not use_land_use_data

        if use_land_use_data:
            self._fix_lu_table(arguments)

    def _fix_lu_table(self, arguments: list[Argument]) -> None:
        """Fixes lu table so 'LUCODE Field' column has valid choices for the shapefile (or no choices if it's a raster).

        Args:
            arguments: The tool arguments.
        """
        lu_table_df = arguments[ARG_INPUT_LAND_USE_TABLE].value
        for _index, row in lu_table_df.iterrows():
            if row[mc.LU_COL_TYPE] == mc.RASTER:
                row[mc.LU_COL_CHOICES] = []  # LUCODE field is only for shapefiles
                row[mc.LU_COL_LU_CODE] = ''  # LUCODE field is only for shapefiles
            else:  # shapefile
                file_path = row[mc.LU_COL_FILE]
                if not file_path:
                    continue

                if file_path in self._lu_shp_fields and row[mc.LU_COL_CHOICES] != self._lu_shp_fields:
                    field_names = self._lu_shp_fields[file_path]
                    row[mc.LU_COL_CHOICES] = field_names
                    if row[mc.LU_COL_LU_CODE] not in field_names:
                        row[mc.LU_COL_LU_CODE] = 'LUCODE' if 'LUCODE' in field_names else field_names[0]
                else:
                    # Read the fields in the shapefile, put them in the table, and save them for next time
                    try:
                        vi = VectorInput(file_path)
                    except ValueError:
                        row[mc.LU_COL_CHOICES] = []
                        row[mc.LU_COL_LU_CODE] = ''  # LUCODE field is only for shapefiles
                        continue

                    # Get field names and make them the choices
                    field_names = [field.name for field in vi.get_fields()]
                    self._lu_shp_fields[file_path] = field_names
                    row[mc.LU_COL_CHOICES] = field_names
                    row[mc.LU_COL_LU_CODE] = 'LUCODE' if 'LUCODE' in field_names else field_names[0]

    def validate_arguments(self, arguments: list[Argument]) -> dict[str, str]:
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors: dict[str, str] = {}

        # Make sure we're doing soils and/or land use
        if not arguments[ARG_INPUT_USE_SOIL_DATA].value and not arguments[ARG_INPUT_USE_LAND_USE_DATA].value:
            errors[arguments[ARG_INPUT_USE_SOIL_DATA].name] = 'You must use soil data or land use data or both.'

        # Validate soil data
        if arguments[ARG_INPUT_USE_SOIL_DATA].value:
            if not _validate_soil_table(arguments, errors):
                return errors

        # Validate land use data
        if arguments[ARG_INPUT_USE_LAND_USE_DATA].value:
            if not _validate_lu_table(arguments, errors):
                return errors

        return errors

    def _get_tool_inputs(self, arguments: list[Argument]) -> mc.ToolInputs:
        """Gets the tool arguments that we need."""
        tool_inputs = mc.ToolInputs()
        tool_inputs.mapping_table_name = self.mapping_table_name
        tool_inputs.co_grid = self.co_grid
        tool_inputs.ugrid = self.ugrid
        tool_inputs.ugrid_node = self.ugrid_node
        tool_inputs.use_soil_data = arguments[ARG_INPUT_USE_SOIL_DATA].value
        if tool_inputs.use_soil_data:
            tool_inputs.soil_table = arguments[ARG_INPUT_SOIL_TABLE].value
        tool_inputs.use_land_use_data = arguments[ARG_INPUT_USE_LAND_USE_DATA].value
        if tool_inputs.use_land_use_data:
            tool_inputs.land_use_table = arguments[ARG_INPUT_LAND_USE_TABLE].value
        tool_inputs.index_map_name = arguments[ARG_INPUT_INDEX_MAP_NAME].value
        tool_inputs.wkt = self.default_wkt
        tool_inputs.vertical_units = self.vertical_units
        return tool_inputs

    def run(self, arguments: list[Argument]) -> None:
        """Override to run the tool.

        Args:
            arguments: The tool arguments.
        """
        tool_inputs = self._get_tool_inputs(arguments)
        self.run_results = mc.run(tool_inputs, self.logger)
        if self.run_results.index_map_dataset:
            self.set_output_dataset(self.run_results.index_map_dataset)


def _create_soils_table() -> tuple[TableDefinition, pd.DataFrame]:
    """Creates and returns a TableDefinition and an empty pd.DataFrame."""
    shp_filter = 'Shapefiles (*.shp);; All files (*.*)'
    columns = [
        InputFileColumnType(header=mc.SOIL_COL_FILE, tool_tip='Shapefile path', file_filter=shp_filter),
    ]
    table_def = TableDefinition(columns)
    return table_def, table_def.to_pandas()


def _create_land_use_table() -> tuple[TableDefinition, pd.DataFrame]:
    """Creates and returns a TableDefinition and an empty pd.DataFrame."""
    filter = 'Land use files (*.shp *.tif);; Shapefiles (*.shp);; Raster files (*.tif);; All files (*.*)'
    csv_filter = 'Comma separated values (CSV) files (*.csv *.txt);; All files (*.*)'
    choices = [mc.SHAPEFILE, mc.RASTER]
    type_desc = 'File type: shapefile or raster'
    tex_field_desc = 'LUCODE field (only for shapefiles)'
    columns = [
        StringColumnType(header=mc.LU_COL_TYPE, tool_tip=type_desc, default=mc.RASTER, choices=choices),
        InputFileColumnType(header=mc.LU_COL_FILE, tool_tip='File path', file_filter=filter),
        InputFileColumnType(header=mc.LU_COL_CSV_FILE, tool_tip='File path of CSV file', file_filter=csv_filter),
        StringColumnType(header=mc.LU_COL_LU_CODE, tool_tip=tex_field_desc, default='LUCODE', choices=CHOICES_COL),
        ChoicesColumnType(header=mc.LU_COL_CHOICES, tool_tip='', default=['']),
    ]
    table_def = TableDefinition(columns)
    return table_def, table_def.to_pandas()


def _validate_soil_table(arguments: list[Argument], errors: dict[str, str]) -> bool:
    """Validates the soil shapefiles."""
    soil_table_df = arguments[ARG_INPUT_SOIL_TABLE].value
    key = arguments[ARG_INPUT_SOIL_TABLE].name

    # Verify the files exist and have the "TEXTURE" field
    valid_row_exists = False
    for index, row in soil_table_df.iterrows():

        # Make sure the file has been specified
        file_path = row[mc.SOIL_COL_FILE]
        if not file_path:
            errors[key] = f'Soil shapefile in row {index} has not been specified.'
            return False

        # Make sure file exists
        if not file_path or not Path(file_path).exists():
            errors[key] = f'Soil shapefile "{file_path}" in row {index} does not exist.'
            return False

        # Make sure it's a valid shapefile
        vi = _open_shapefile(file_path, index, key, errors)
        if vi is None:
            return False

        # Make sure it's a polygon shapefile
        if not _is_polygon_shapefile(vi, file_path, index, key, errors):
            return False

        # Make sure it has the "TEXTURE" field
        field_names = {field.name for field in vi.get_fields()}
        if 'TEXTURE' not in field_names:
            errors[key] = f'Shapefile "{file_path}" in row {index} does not have a "TEXTURE" field.'
            return False

        valid_row_exists = True

    # Check if no files were specified
    if not valid_row_exists:
        errors[key] = '"Use soil data" selected but no soil shapefiles specified.'
        return False

    return True


def _validate_lu_table(arguments: list[Argument], errors: dict[str, str]) -> bool:
    """Validates the land use table."""
    lu_table_df = arguments[ARG_INPUT_LAND_USE_TABLE].value
    key = arguments[ARG_INPUT_LAND_USE_TABLE].name

    valid_row_exists = False
    for index, row in lu_table_df.iterrows():

        # Make sure the file has been specified
        file_path = row[mc.LU_COL_FILE]
        if not file_path:
            errors[key] = f'Land use file in row {index} has not been specified.'
            return False

        # Make sure file exists
        if not file_path or not Path(file_path).exists():
            errors[key] = f'Land use file "{file_path}" in row {index} does not exist.'
            return False

        # Make sure it's a valid shapefile or raster
        if row[mc.LU_COL_TYPE] == mc.SHAPEFILE:

            # Make sure it's a valid shapefile
            vi = _open_shapefile(file_path, index, key, errors)
            if vi is None:
                return False

            # Make sure it's a polygon shapefile
            if not _is_polygon_shapefile(vi, file_path, index, key, errors):
                return False

        else:
            try:
                RasterInput(file_path)
            except ValueError:
                errors[key] = f'Could not open "{file_path}" in row {index} as a raster. Check the file type.'
                return False

        # Make sure the CSV file has been specified
        csv_file_path = row[mc.LU_COL_CSV_FILE]
        if not csv_file_path:
            errors[key] = f'CSV file in row {index} has not been specified.'
            return False

        # Make sure the CSV file exists
        if not Path(csv_file_path).exists():
            errors[key] = f'CSV file "{csv_file_path}" in row {index} does not exist.'
            return False

        valid_row_exists = True

    # Check if no files were specified
    if not valid_row_exists:
        errors[key] = '"Use land use data" selected but no land use files specified.'
        return False


def _open_shapefile(file_path: str, row: int, key: str, errors: dict[str, str]) -> 'VectorInput | None':
    """Returns a VectorInput if the file can be opened as a shapefile, else returns None and adds to errors.

    Args:
        file_path: File path.
        row: Table row (1-based).
        key: key added to errors, along with the error string, if file couldn't be opened as a shapefile.
        errors: Errors dict. If file couldn't be opened as a shapefile, an error string is added using key.

    Returns:
        See description.
    """
    try:
        vi = VectorInput(file_path)
        return vi
    except ValueError:
        errors[key] = f'Could not open "{file_path}" in row {row} as a shapefile. Check the file type.'
        return None


def _is_polygon_shapefile(vi: VectorInput, file_path: str, row: int, key: str, errors: dict[str, str]) -> bool:
    """Returns True if the file is a polygon shapefile, else returns False and adds to errors.

    Args:
        vi: VectorInput object.
        file_path: File path.
        row: Table row (1-based).
        key: key added to errors, along with the error string, if file couldn't be opened as a shapefile.
        errors: Errors dict. If file couldn't be opened as a shapefile, an error string is added using key.

    Returns:
        See description.
    """
    if not tool_util.is_polygon_shapefile(vi):
        errors[key] = f'"{file_path}" in row {row} is not a polygon shapefile.'
        return False
    return True
