"""Dialog for showing the map locator."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import sys

# 2. Third party modules
from jinja2 import Template
from PySide2 import QtCore
from PySide2.QtWebChannel import QWebChannel
from PySide2.QtWebEngineWidgets import QWebEnginePage, QWebEngineView
from PySide2.QtWidgets import QApplication

# 3. Aquaveo modules

# 4. Local modules


class Element(QtCore.QObject):
    """Base class for constructing a javascript call that has a callback."""
    def __init__(self, name, map_name, parent=None):
        """Initialization function for Element.

        Args:
            name: javascript function name
            map_name: Map name
            parent (:obj:`QWidget`): Parent window
        """
        super(Element, self).__init__(parent)
        self._name = name
        self._map_name = map_name
        self._parent = parent

    @property
    def name(self):
        """Returns the javascript function name."""
        return self._name

    @property
    def map_name(self):
        """Returns the folium map name."""
        return self._map_name

    def script(self):
        """Not implemented."""
        raise NotImplementedError


class ParametersObject(Element):
    """Class for constructing a javascript call to get map bounding box and other parameters."""
    def script(self):
        """Javascript script to get the window bounds."""
        _script = r"""
        function windowInfo(){
            var i = 0;
            var id_timer = setInterval(function() { //arbitrary delay
                i++
                if(i>2)
                {
                    var bounds = {{map_name}}.getBounds();
                    var center = bounds.getCenter();
                    {{name}}.set_parameters(center.lat, center.lng, {{map_name}}.getZoom(), bounds.getWest(),
                                            bounds.getEast(), bounds.getSouth(), bounds.getNorth());
                }
            }, 100);
        }
        windowInfo();
        """
        return Template(_script).render(name=self.name, map_name=self.map_name)

    @QtCore.Slot(float, float, int, float, float, float, float)
    def set_parameters(self, lat, lon, zoom, west, east, south, north):
        """Forwards to the window set parameters function."""
        self._parent.set_parameters(lat, lon, zoom, west, east, south, north)


class WebEnginePage(QWebEnginePage):
    """Derived class for a QWebEnginePage that is used to return Javascript data (used in ParametersWebPage)."""
    def __init__(self, *args, **kwargs):
        """Initialization function for the Web Engine Page."""
        super(WebEnginePage, self).__init__(*args, **kwargs)
        self.loadFinished.connect(self.on_load_finished)
        self._objects = []

    def add_object(self, obj):
        """Adds a javascript object to the web page."""
        self._objects.append(obj)

    @QtCore.Slot(bool)
    def on_load_finished(self, ok):
        """Called when the page finishes loading."""
        if ok:
            self._load_qwebchannel()
            self._load_objects()

    def _load_qwebchannel(self):
        file = QtCore.QFile(":/qtwebchannel/qwebchannel.js")
        if file.open(QtCore.QIODevice.ReadOnly):
            content = file.readAll()
            file.close()
            self.runJavaScript(content.data().decode())
        if self.webChannel() is None:
            channel = QWebChannel(self)
            self.setWebChannel(channel)

    def _load_objects(self):
        if self.webChannel() is not None:
            objects = {obj.name: obj for obj in self._objects}
            self.webChannel().registerObjects(objects)
            _script = r"""
            {% for obj in objects %}
            var {{obj}};
            {% endfor %}
            new QWebChannel(qt.webChannelTransport, function (channel) {
            {% for obj in objects %}
                {{obj}} = channel.objects.{{obj}};
            {% endfor %}
            });
            """
            self.runJavaScript(Template(_script).render(objects=objects.keys()))
            for obj in self._objects:
                if isinstance(obj, Element):
                    self.runJavaScript(obj.script())


class ParametersWebPage(QWebEngineView):
    """Derived class for a QWebEngineView that is used to return Javascript data."""
    def __init__(self, parent=None):
        """Initialization function for the Web Engine View."""
        super().__init__(parent)
        page = WebEnginePage(self)
        self.setPage(page)


if __name__ == "__main__":
    # Create the Qt Application
    app = QApplication(sys.argv)
    # Create and show the form
    web = ParametersWebPage()
    web.show()
    # Run the main Qt loop
    sys.exit(app.exec_())
