"""Point symbol display options."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field
from enum import IntEnum

# 2. Third party modules
from PySide2.QtGui import QColor

# 3. Aquaveo modules

# 4. Local modules


class PointSymbol(IntEnum):
    """An enumeration matching the one in XMS.

    Values not supported in XMS are commented out for the time being.
    """
    PT_SQUARE = 0
    PT_SQUARE_H = 1
    PT_CIRCLE = 2
    PT_CIRCLE_H = 3
    PT_TRI = 4
    PT_TRI_H = 5
    PT_DIAMOND = 6
    PT_DIAMOND_H = 7
    PT_CROSS = 8
    PT_X = 9
    PT_SURVEY = 10
    # the following are not supported
    # PT_WSE = 11
    # PT_GAUGE = 12
    # PT_FLOWER = 13
    # PT_STAR = 14
    # PT_SMILEY = 15
    # PT_SKULL = 16
    # PT_NUM = 17


@dataclass
class PointOptions:
    """A class to hold the display options."""
    color: QColor = field(default_factory=lambda: QColor('black'))  # To instantiate distinct black QColor objects
    size: int = 7
    symbol: PointSymbol = PointSymbol.PT_CIRCLE

    def from_dict(self, opt_dict):
        """Populate options from a dict.

        Args:
            opt_dict (dict): dict of the point symbol display option attributes
        """
        red = 0
        green = 0
        blue = 0
        alpha = 0
        if 'red' in opt_dict:
            red = opt_dict['red']
        if 'green' in opt_dict:
            green = opt_dict['green']
        if 'blue' in opt_dict:
            blue = opt_dict['blue']
        if 'alpha' in opt_dict:
            alpha = opt_dict['alpha']
        self.color = QColor(red, green, blue, alpha)
        if 'size' in opt_dict:
            self.size = opt_dict['size']
        if 'symbol' in opt_dict:
            self.symbol = PointSymbol(int(opt_dict['symbol']))

    def __iter__(self):
        """Used for conversion to dict."""
        yield 'red', self.color.red()  # Decompose QColor to R,G,B,A components
        yield 'green', self.color.green()
        yield 'blue', self.color.blue()
        yield 'alpha', self.color.alpha()
        yield 'size', self.size
        yield 'symbol', int(self.symbol)  # Cast enum to int for sending to C++
