"""Dialog for viewing and editing display options in XMS."""

# 1. Standard Python modules
import os
from typing import Optional
import webbrowser

# 2. Third party modules
import orjson
from PySide2.QtCore import Qt, Slot
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.dialogs.category_display_options_list_ui import Ui_CategoryDisplayOptionsDialog
from xms.guipy.dialogs.help_getter import HelpGetter
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.settings import SettingsManager
from xms.guipy.time_format import ISO_DATETIME_FORMAT


class CategoryDisplayOptionsDialog(XmsDlg):
    """A dialog to let the user set point display options."""
    def __init__(
        self,
        category_lists: list[CategoryDisplayOptionList],
        parent: Optional[QWidget] = None,
        change_all_color: bool = False,
        show_on_off: bool = True,
        show_targets: bool = False,
        package_name: str = '',
        registry_keys: Optional[list[str]] = None,
        **kwargs
    ):
        """Initializes the dialog.

        Args:
            category_lists: The categories and the target type of the list.
            parent: The parent window.
            change_all_color: If True, the same color will be assigned to all categories when the change
                all button is pushed.
            show_on_off: If False, the checkboxes column and the 'All On', 'All Off' buttons are
                hidden.
            show_targets: If True, widgets for calibration targets will be shown.
            package_name: The name of the package. Used for storing to registry.
            registry_keys: The keys to use for storing the options to registry. Must be one per
                category display list in the dialog, and they must be in the same order as the display lists.

                Note that if both 'package_name' and 'registry_key' kwargs are defined, a button to save the current
                options as defaults in the registry will be added.
        """
        super().__init__(parent, 'xms.guipy.dialogs.category_display_options_list.CategoryDisplayOptionsDialog')
        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)

        self.help_url = "https://www.xmswiki.com/wiki/SMS:Display_Options"
        self.help_getter: HelpGetter | None = None
        self.read_only = False  # See set_read_only. If true, dialog cannot be edited.

        self.ui = Ui_CategoryDisplayOptionsDialog()
        self.ui.setupUi(self)
        self.ui.display_options.change_all_color = change_all_color
        self.ui.display_options.set_show_on_off(show_on_off)
        self.ui.display_options.set_show_targets(show_targets)
        self.ui.display_options.set_category_lists(category_lists)

        self.ui.txt_read_only.setStyleSheet("QLabel { background-color : rgb(255, 255, 128); color : rgb(0, 0, 0); }")
        self.set_read_only(False)

        # Get registry values
        self._package_name = package_name
        self._reg_keys = registry_keys or []
        if self._package_name and self._reg_keys:
            self.ui.btn_save_default.clicked.connect(self.on_save_default)
        else:
            self.ui.btn_save_default.hide()

    def set_read_only(self, read_only):
        """Set the dialog to be read-only.

        Args:
            read_only (bool): True if the dialog should not be editable
        """
        self.read_only = read_only
        self.ui.txt_read_only.setVisible(self.read_only)
        self.ui.display_options.setEnabled(not self.read_only)

    def get_category_lists(self):
        """Returns the current category display options lists.

        Returns:
            cat_list (CategoryDisplayOptionList): The current category display options list.
        """
        return self.ui.display_options.get_category_lists()

    def on_save_default(self):
        """Save the current options to the registry."""
        display_lists = self.get_category_lists()
        if len(display_lists) != len(self._reg_keys):
            raise RuntimeError('Unexpected number of display option lists.')

        settings = SettingsManager()
        for display_list, reg_key in zip(display_lists, self._reg_keys):
            json_dict = {
                'target_type': int(display_list.target_type),
                'is_ids': int(display_list.is_ids),
                'show_targets': int(display_list.show_targets),
                'logarithmic_scale': int(display_list.logarithmic_scale),
                'target_scale': display_list.target_scale,
                # Not sure if it really makes sense to store a default reference date in the registry, probably
                # will be set by a script if it is really being used (not hidden in the dialog).
                'obs_reftime': display_list.obs_reftime.strftime(ISO_DATETIME_FORMAT),
                # 'obs_dset_uuid' - don't store this in the registry
                'enable_polygon_labels': int(display_list.enable_polygon_labels),
                'categories': list(display_list),
            }
            json_text = orjson.dumps(json_dict)
            settings.save_setting(self._package_name, reg_key, json_text)
        app_name = os.environ.get('XMS_PYTHON_APP_NAME', '')
        msg = 'Successfully saved current display options as the defaults.'
        message_with_ok(self, message=msg, app_name=app_name, icon='Information', win_icon=self.windowIcon())

    @Slot()
    def help_requested(self):
        """Called when the Help button is clicked."""
        if self.help_getter:
            super().help_requested()
        else:
            webbrowser.open(self.help_url)
