"""The table and plots widget."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import sys

# 2. Third party modules
from matplotlib.backends.backend_qt5agg import FigureCanvas, NavigationToolbar2QT as NavigationToolbar
from matplotlib.figure import Figure
from matplotlib.ticker import ScalarFormatter
import numpy as np
from PySide2.QtCore import QSize, Qt
from PySide2.QtWidgets import (
    QCheckBox, QGroupBox, QHBoxLayout, QLabel, QLineEdit, QListWidget, QListWidgetItem, QPushButton, QScrollArea,
    QTabWidget, QVBoxLayout, QWidget
)

# 3. Aquaveo modules
from xms.guipy.data.plot_and_table_data_base import PlotsAndTableDataBase
from xms.guipy.dialogs import message_box
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets.qx_table_view import QxTableView

# 4. Local modules


def np_arrays_from_file(file_name):
    """Read data in the files into numpy arrays.

    Args:
        file_name (str): file name

    Returns:
        (tuple(list[array_names], numpy 2d array)): output
    """
    with open(file_name, 'r') as f:
        lines = f.readlines()

    # remove lines that start with '//'
    while lines[0].startswith('//') and not lines[0].startswith('//    Time(hr)'):
        lines.pop(0)
    if lines[0].startswith('//'):
        lines[0] = lines[0][2:]
    headings = lines[0].split()
    # remove these columns from monitor point files because this data doesn't make sense
    if len(headings) > 0 and headings[-1] == 'CONC_T_ppm':
        headings.pop()
    if len(headings) > 0 and 'ERO_DEP_' in headings[-1]:
        headings.pop()

    arrays = [np.zeros(len(lines) - 1) for _ in range(len(headings))]
    for i in range(1, len(lines)):
        vals = lines[i].split()
        nvals = min(len(vals), len(arrays))
        for j in range(nvals):
            arrays[j][i - 1] = float(vals[j])

    # point sediment file
    fname = file_name[:-4] + '_SED.dat'
    if os.path.isfile(fname):
        sed_headings, sed_arrays = np_arrays_from_file(fname)
        headings = headings + sed_headings[1:]
        arrays = arrays + sed_arrays[1:]

    return headings, arrays


class PlotsAndTableWidget(QWidget):
    """A dialog for showing SRH-2D Output plots."""
    def __init__(self, parent, data: PlotsAndTableDataBase):
        """Initializes the dialog, sets up the ui.

        Args:
            parent (QWidget): Parent dialog
            data (PlotsAndTableDataBase): data class to populate plot
        """
        super().__init__(parent)
        self.widgets = {}

        if data is not None:
            if not isinstance(data, PlotsAndTableDataBase):
                raise TypeError("data must be an instance of PlotsAndTableDataBase or its subclasses")
            self._data = data
        else:
            self._data = None

        self._checked_sims = []
        self._max_x: float = sys.float_info.max
        self._max_y: float = sys.float_info.max
        self._min_y: float = -sys.float_info.max
        self._log_min_y: float = sys.float_info.min  # Smallest positive float
        self._log_max_y: float = sys.float_info.max
        self._last_min_y: float = self._min_y
        self._last_max_y: float = self._max_y
        self._last_log_min_y: float = self._log_min_y
        self._last_log_max_y: float = self._log_max_y

        self._update_min_max_plot_values()
        self._updating_time_range = False
        self._updating_value_range = False

        self.figure = None  # matplotlib.figure Figure
        self.canvas = None  # matplotlib.backends.backend_qt5agg FigureCanvas
        self.ax = None  # matplotlib Axes
        self.toolbar = None  # matplotlib navigation toolbar

        self._setup_ui()

        # set up the initial state of the dialog
        self._on_list_state_change()
        self._on_min_max_x_changed()
        self._on_min_max_y_changed()

    def _update_min_max_plot_values(self):
        """Updates the min and max plot values."""
        self._max_x = self._data.get_plot_max_x()
        self._min_y, self._max_y, self._log_min_y, self._log_max_y = self._data.get_plot_min_max_y()

        self._last_log_min_y, self._last_log_max_y = self._log_min_y, self._log_max_y
        self._set_x_axis_validators_on_edit_fields()
        self._set_y_axis_validators_on_edit_fields()

    def _set_x_axis_validators_on_edit_fields(self):
        """Create validators with updated min and max values."""
        val = QxDoubleValidator(parent=self, bottom=0.0, top=self._max_x)
        if 'min_x_edit' in self.widgets:
            self.widgets['min_x_edit'].setValidator(val)
        if 'max_x_edit' in self.widgets:
            self.widgets['max_x_edit'].setValidator(val)

    def _set_y_axis_validators_on_edit_fields(self):
        """Create validators with updated min and max values."""
        log_scale = False
        if 'y_axis_log_scale' in self.widgets:
            log_scale = self.widgets['y_axis_log_scale'].isChecked()
        min_y = self._min_y if not log_scale else self._log_min_y
        max_y = self._max_y if not log_scale else self._log_max_y
        val_1 = QxDoubleValidator(parent=self, bottom=min_y, top=max_y)
        if 'min_y_edit' in self.widgets:
            self.widgets['min_y_edit'].setValidator(val_1)
        if 'max_y_edit' in self.widgets:
            self.widgets['max_y_edit'].setValidator(val_1)

    def _setup_ui(self) -> None:
        """Sets up the dialog controls."""
        # Make the main layout a vertical layout
        self.widgets['main_vert_layout'] = QVBoxLayout()
        self.setLayout(self.widgets['main_vert_layout'])

        self._setup_main_warning()
        self._setup_left_and_right_layouts()

        # left side of the dialog
        self._setup_feature_ids_list()
        self._setup_simulations_list()
        self._setup_plots_list()
        self._setup_x_axis_widgets()
        self._setup_y_axis_widgets()

        # right side of the dialog
        self._setup_tab_widget()
        self._setup_plot_tab()
        self._setup_table_tabs()

        # bottom of the dialog
        self._setup_coverage_edited_warning()

    def _setup_coverage_edited_warning(self) -> None:
        """Sets up the warning about the coverage being edited."""
        msg = self._data.get_bottom_warning()
        self.widgets['warning_label'] = QLabel(msg)
        self.widgets['warning_label'].setWordWrap(True)
        self.widgets['main_vert_layout'].addWidget(self.widgets['warning_label'])

    def _setup_table_tabs(self) -> None:
        """Sets up the table tabs."""
        # Table
        self.widgets['tables'] = []
        for tab in self.widgets['table_tabs']:
            self.widgets['tables'].append(QxTableView())
            tab.layout = QVBoxLayout()
            tab.layout.addWidget(self.widgets['tables'][-1])
            tab.setLayout(tab.layout)

    def _setup_plot_tab(self) -> None:
        """Sets up the plot tab."""
        # Plot
        self.figure = Figure()
        # self.figure.set_tight_layout(True)  # Frames the plots
        self.figure.set_layout_engine(layout='tight')
        self.canvas = FigureCanvas(self.figure)
        self.canvas.sizeHint = lambda: QSize(150, 150)  # needed so that the vertical can be shrunk
        self.canvas.setMinimumWidth(150)  # So user can't resize it to nothing
        self.widgets['plot_tab'].layout = QVBoxLayout()
        self.toolbar = NavigationToolbar(self.canvas, self)
        # Remove the "Configure subplots" button. We aren't really sure what this does.
        for x in self.toolbar.actions():
            if x.text() == 'Subplots':
                self.toolbar.removeAction(x)
        self.widgets['plot_tab'].layout.addWidget(self.toolbar)
        self.widgets['plot_tab'].layout.addWidget(self.canvas)
        self.widgets['plot_tab'].setLayout(self.widgets['plot_tab'].layout)
        self.widgets['right_vert_layout'].addWidget(self.widgets['right_tab_widgets'])
        self.ax = self.figure.add_subplot(111)

    def _setup_tab_widget(self) -> None:
        """Sets up the tab widget."""
        self.widgets['right_tab_widgets'] = QTabWidget()
        self.widgets['plot_tab'] = QWidget()
        self.widgets['right_tab_widgets'].addTab(self.widgets['plot_tab'], 'Plot')
        self.widgets['table_tabs'] = []
        for f_id in self._data.fids:
            for sim in self._data.simulations:
                self.widgets['table_tabs'].append(QWidget())
                f_label = self._data.get_f_label(f_id)
                label = f'Table: {f_label} - {sim}'
                self.widgets['right_tab_widgets'].addTab(self.widgets['table_tabs'][-1], label)

    def _setup_y_axis_widgets(self) -> None:
        """Sets up the y-axis widgets."""
        # value range controls
        self.widgets['value_range_group'] = QGroupBox('Y axis')
        self.widgets['value_range_group'].setMaximumWidth(200)
        self.widgets['left_vert_layout'].addWidget(self.widgets['value_range_group'])
        self.widgets['value_range_vlayout'] = QVBoxLayout()
        self.widgets['value_range_group'].setLayout(self.widgets['value_range_vlayout'])
        self.widgets['y_axis_specify_range'] = QCheckBox('Specify range')
        self.widgets['y_axis_specify_range'].stateChanged.connect(self._on_min_max_y_changed)
        self.widgets['value_range_vlayout'].addWidget(self.widgets['y_axis_specify_range'])
        self.widgets['min_y_label'] = QLabel('Minimum value:')
        self.widgets['value_range_vlayout'].addWidget(self.widgets['min_y_label'])
        self.widgets['min_y_edit'] = QLineEdit(str(self._min_y))
        self.widgets['min_y_edit'].editingFinished.connect(self._on_min_max_y_changed)
        self.widgets['value_range_vlayout'].addWidget(self.widgets['min_y_edit'])
        self.widgets['max_y_label'] = QLabel('Maximum value:')
        self.widgets['value_range_vlayout'].addWidget(self.widgets['max_y_label'])
        self.widgets['max_y_edit'] = QLineEdit(str(self._max_y))
        self.widgets['max_y_edit'].editingFinished.connect(self._on_min_max_y_changed)
        self.widgets['value_range_vlayout'].addWidget(self.widgets['max_y_edit'])
        self.widgets['left_vert_layout'].addStretch()
        self._set_y_axis_validators_on_edit_fields()

        self.widgets['y_axis_log_scale'] = QCheckBox('Log scale')
        self.widgets['y_axis_log_scale'].stateChanged.connect(self._on_log_scale)
        self.widgets['value_range_vlayout'].addWidget(self.widgets['y_axis_log_scale'])

    def _setup_x_axis_widgets(self) -> None:
        """Sets up the x-axis widgets."""
        self.widgets['time_range_group'] = QGroupBox('X axis')
        self.widgets['time_range_group'].setMaximumWidth(200)
        self.widgets['left_vert_layout'].addWidget(self.widgets['time_range_group'])
        self.widgets['time_range_vlayout'] = QVBoxLayout()
        self.widgets['time_range_group'].setLayout(self.widgets['time_range_vlayout'])
        self.widgets['x_axis_specify_range'] = QCheckBox('Specify range')
        self.widgets['x_axis_specify_range'].stateChanged.connect(self._on_min_max_x_changed)
        self.widgets['time_range_vlayout'].addWidget(self.widgets['x_axis_specify_range'])
        self.widgets['min_x_label'] = QLabel('Minimum time:')
        self.widgets['time_range_vlayout'].addWidget(self.widgets['min_x_label'])
        self.widgets['min_x_edit'] = QLineEdit('0.0')
        self.widgets['min_x_edit'].editingFinished.connect(self._on_min_max_x_changed)
        self.widgets['time_range_vlayout'].addWidget(self.widgets['min_x_edit'])
        self.widgets['max_x_label'] = QLabel('Maximum time:')
        self.widgets['time_range_vlayout'].addWidget(self.widgets['max_x_label'])
        self.widgets['max_x_edit'] = QLineEdit(str(self._max_x))
        self.widgets['max_x_edit'].editingFinished.connect(self._on_min_max_x_changed)
        self.widgets['time_range_vlayout'].addWidget(self.widgets['max_x_edit'])
        self._set_x_axis_validators_on_edit_fields()

    def _setup_plots_list(self) -> None:
        """Sets up the list of plots."""
        # list control for plot types
        self.widgets['plot_type_label'] = QLabel('Plots:')
        self.widgets['left_vert_layout'].addWidget(self.widgets['plot_type_label'])
        self.widgets['plot_type_list'] = QListWidget()
        self.widgets['plot_type_list'].setMaximumWidth(200)
        self.widgets['plot_type_list'].itemChanged.connect(self._on_list_state_change)
        for plot in self._data.plot_list:
            item = QListWidgetItem(plot)
            item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
            item.setCheckState(Qt.Unchecked)
            if plot.startswith(self._data.default_plot_on):
                item.setCheckState(Qt.Checked)
            self.widgets['plot_type_list'].addItem(item)
        self.widgets['left_vert_layout'].addWidget(self.widgets['plot_type_list'])

    def _setup_simulations_list(self) -> None:
        """Sets up the list of simulations."""
        self.widgets['simulation_label'] = QLabel('Simulations:')
        self.widgets['left_vert_layout'].addWidget(self.widgets['simulation_label'])
        self.widgets['simulation_list'] = QListWidget()
        self.widgets['simulation_list'].setMaximumWidth(200)
        self.widgets['simulation_list'].itemChanged.connect(self._on_list_state_change)
        for sim in self._data.simulations:
            item = QListWidgetItem(sim)
            item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
            item.setCheckState(Qt.Checked)
            self.widgets['simulation_list'].addItem(item)
        self.widgets['left_vert_layout'].addWidget(self.widgets['simulation_list'])
        self.widgets['button_horiz_layout'] = QHBoxLayout()
        self.widgets['left_vert_layout'].addLayout(self.widgets['button_horiz_layout'])
        self.widgets['sim_all_on_btn'] = QPushButton('All on')
        self.widgets['sim_all_on_btn'].clicked.connect(self._on_btn_sim_all_on)
        self.widgets['button_horiz_layout'].addWidget(self.widgets['sim_all_on_btn'])
        self.widgets['sim_all_off_btn'] = QPushButton('All off')
        self.widgets['sim_all_off_btn'].clicked.connect(self._on_btn_sim_all_off)
        self.widgets['button_horiz_layout'].addWidget(self.widgets['sim_all_off_btn'])

    def _setup_feature_ids_list(self) -> None:
        """Sets up the list of feature ids."""
        if self._data.feature_ids:
            self.widgets['feature_ids_label'] = QLabel(f'{self._data.feature_type} ids:')
            self.widgets['left_vert_layout'].addWidget(self.widgets['feature_ids_label'])
            self.widgets['feature_ids_list'] = QListWidget()
            self.widgets['feature_ids_list'].setMaximumWidth(200)
            self.widgets['feature_ids_list'].itemChanged.connect(self._on_list_state_change)
            for fid in self._data.feature_ids:
                item = QListWidgetItem(f'{fid}')
                item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
                item.setCheckState(Qt.Checked)
                self.widgets['feature_ids_list'].addItem(item)
            self.widgets['left_vert_layout'].addWidget(self.widgets['feature_ids_list'])

    def _setup_left_and_right_layouts(self) -> None:
        """Sets up the left and right layouts."""
        # horizontal layout
        self.widgets['main_horiz_layout'] = QHBoxLayout()
        self.widgets['main_vert_layout'].addLayout(self.widgets['main_horiz_layout'])
        # 2 vertical layouts 1 for list box and time range controls and 2 for the plot on the right
        self.widgets['left_layout'] = QVBoxLayout()
        self.widgets['main_horiz_layout'].addLayout(self.widgets['left_layout'], stretch=1)
        self.widgets['left_scroll_area'] = QScrollArea(self)
        self.widgets['left_scroll_area'].setWidgetResizable(True)
        self.widgets['left_scroll_area'].setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        self.widgets['left_scroll_area'].setMaximumWidth(200)
        self.widgets['left_scroll_area'].setMinimumWidth(200)
        self.widgets['left_layout'].addWidget(self.widgets['left_scroll_area'])
        self.widgets['left_scroll_area_widget'] = QWidget()
        self.widgets['left_scroll_area'].setWidget(self.widgets['left_scroll_area_widget'])
        self.widgets['left_vert_layout'] = QVBoxLayout()
        self.widgets['left_scroll_area_widget'].setLayout(self.widgets['left_vert_layout'])

        self.widgets['main_horiz_layout'].addLayout(self.widgets['left_vert_layout'], stretch=1)
        self.widgets['right_vert_layout'] = QVBoxLayout()
        self.widgets['main_horiz_layout'].addLayout(self.widgets['right_vert_layout'])

    def _setup_main_warning(self):
        """Sets up the main warning."""
        main_warning = self._data.get_main_warning()
        self.widgets['main_warning'] = QLabel(main_warning)
        self.widgets['main_warning'].setStyleSheet('font-weight: bold; color: red')
        show_main_warning = False
        if len(self._data.simulations) < 1 or len(self._data.plot_list) < 1:
            show_main_warning = True
        if show_main_warning:
            self.widgets['main_vert_layout'].addWidget(self.widgets['main_warning'])
        return show_main_warning

    def _on_min_max_x_changed(self):
        """Called when the min or max time has changed."""
        if self._updating_time_range:
            return

        # make sure the min/max are not inverted
        min_x = float(self.widgets['min_x_edit'].text())
        max_x = float(self.widgets['max_x_edit'].text())
        if min_x > max_x:
            self._updating_time_range = True
            self.widgets['min_x_edit'].setText(str(max_x))
            self.widgets['max_x_edit'].setText(str(min_x))
            self._updating_time_range = False

        # enabling/disabling
        self.widgets['min_x_label'].setEnabled(self.widgets['x_axis_specify_range'].isChecked())
        self.widgets['min_x_edit'].setEnabled(self.widgets['x_axis_specify_range'].isChecked())
        self.widgets['max_x_label'].setEnabled(self.widgets['x_axis_specify_range'].isChecked())
        self.widgets['max_x_edit'].setEnabled(self.widgets['x_axis_specify_range'].isChecked())

        self._on_list_state_change()

    def _on_min_max_y_changed(self) -> None:
        """Called when the min or max value has changed."""
        if self._updating_value_range:
            return

        # make sure the min/max are not inverted
        min_y = float(self.widgets['min_y_edit'].text())
        max_y = float(self.widgets['max_y_edit'].text())
        if min_y > max_y:
            self._updating_value_range = True
            self.widgets['min_y_edit'].setText(str(max_y))
            self.widgets['max_y_edit'].setText(str(min_y))
            self._updating_value_range = False

        # enabling/disabling
        self.widgets['min_y_label'].setEnabled(self.widgets['y_axis_specify_range'].isChecked())
        self.widgets['min_y_edit'].setEnabled(self.widgets['y_axis_specify_range'].isChecked())
        self.widgets['max_y_label'].setEnabled(self.widgets['y_axis_specify_range'].isChecked())
        self.widgets['max_y_edit'].setEnabled(self.widgets['y_axis_specify_range'].isChecked())

        self._on_list_state_change()

    def _on_log_scale(self, checked: int) -> None:
        """Called when the log scale checkbox is clicked.

        Args:
            checked: Qt.CheckState (Qt.Unchecked = 0, Qt.PartiallyChecked = 1, Qt.Checked = 2)
        """
        if checked == Qt.Checked:
            self._last_min_y = float(self.widgets['min_y_edit'].text())
            self._last_max_y = float(self.widgets['max_y_edit'].text())
            self.widgets['min_y_edit'].validator().setBottom(self._log_min_y)
            self.widgets['max_y_edit'].validator().setTop(self._log_max_y)
            self.widgets['min_y_edit'].setText(str(self._last_log_min_y))
            self.widgets['max_y_edit'].setText(str(self._last_log_max_y))
        else:
            self._last_log_min_y = float(self.widgets['min_y_edit'].text())
            self._last_log_max_y = float(self.widgets['max_y_edit'].text())
            self.widgets['min_y_edit'].validator().setBottom(self._min_y)
            self.widgets['max_y_edit'].validator().setTop(self._max_y)
            self.widgets['min_y_edit'].setText(str(self._last_min_y))
            self.widgets['max_y_edit'].setText(str(self._last_max_y))
        self._update_plots()

    def _on_list_state_change(self):
        # give message about bad files
        msg = self._data.get_error()
        if msg:
            app_name = os.environ.get('XMS_PYTHON_APP_NAME')
            message_box.message_with_ok(
                parent=self.parent(), message=msg, app_name=app_name, icon='Warning', win_icon=self.windowIcon()
            )

        # get all check features (if applicable)
        self._checked_feature_ids = []
        if self._data.feature_ids:
            for i in range(self.widgets['feature_ids_list'].count()):
                item = self.widgets['feature_ids_list'].item(i)
                if item.checkState() == Qt.Checked:
                    self._checked_feature_ids.append(int(item.text()))

        # get all checked simulations
        self._checked_sims = []
        for i in range(self.widgets['simulation_list'].count()):
            item = self.widgets['simulation_list'].item(i)
            if item.checkState() == Qt.Checked:
                self._checked_sims.append(item.text())

        self._data.checked_plots = []
        for i in range(self.widgets['plot_type_list'].count()):
            item = self.widgets['plot_type_list'].item(i)
            if item.checkState() == Qt.Checked:
                self._data.checked_plots.append(item.text())
        self._update_plots()

    def _update_plots(self):
        """Draws all plots based on selection in the list boxes."""
        self.ax.clear()

        # hide all tables
        for i in range(len(self.widgets['table_tabs'])):
            self.widgets['right_tab_widgets'].setTabVisible(i + 1, False)

        table_tab_index = 0
        show_legend = False
        x_heading = ''
        min_x = None
        max_x = None

        if self.widgets['x_axis_specify_range'].isChecked():
            min_x = float(self.widgets['min_x_edit'].text())
            max_x = float(self.widgets['max_x_edit'].text())

        for f_id in self._data.fids:
            f_label = self._data.get_f_label(f_id)
            self._data.plot_data_files = self._data.feature_ids_plot_data_files[f_id]
            for sim in self._data.simulations:
                table_tab_index += 1
                if self._data.feature_ids and f_id not in self._checked_feature_ids:
                    continue
                if sim not in self._checked_sims:
                    continue

                data, df, update_dlg_min_max = self._data.get_plot_data_and_dataframe(sim, min_x, max_x)
                if data is None or df is None:
                    continue
                if update_dlg_min_max:
                    self._update_min_max_plot_values()

                # Enter data to the plot
                for plot in self._data.checked_plots:
                    plot_idx = self._data.plot_list.index(plot) + 1
                    if plot_idx < len(data[1]):
                        if data[0][plot_idx] != plot:  # Make sure plot and column name match
                            continue
                        label = f'{f_label} - {sim}:{data[0][plot_idx]}'
                        log_scale = self.widgets['y_axis_log_scale'].isChecked()
                        if log_scale:
                            self.ax.plot(data[1][0], np.abs(data[1][plot_idx]), label=label)
                            self.ax.set_yscale('log', nonpositive='mask')
                        else:
                            self.ax.plot(data[1][0], data[1][plot_idx], label=label)
                        show_legend = True
                        if self.widgets['x_axis_specify_range'].isChecked():
                            min_y = float(self.widgets['min_x_edit'].text())
                            max_y = float(self.widgets['max_x_edit'].text())
                            self.ax.set_xbound(min_y, max_y)
                        if self.widgets['y_axis_specify_range'].isChecked():
                            min_y = float(self.widgets['min_y_edit'].text())
                            max_y = float(self.widgets['max_y_edit'].text())
                            self.ax.set_ybound(min_y, max_y)
                        x_heading = data[0][0]

                # Enter data to the tables
                self.widgets['right_tab_widgets'].setTabVisible(table_tab_index, True)
                model = QxPandasTableModel(df)
                self.widgets['tables'][table_tab_index - 1].setModel(model)
                self.widgets['tables'][table_tab_index - 1].resizeColumnsToContents()

        if show_legend:
            self.ax.legend()
        y_formatter = ScalarFormatter(useOffset=False)
        self.ax.yaxis.set_major_formatter(y_formatter)
        self.ax.set_xlabel(self._data.get_x_label(x_heading))
        self.canvas.draw()

    def _on_btn_sim_all_on(self):
        """Signal for when the user clicks the All on button."""
        for i in range(self.widgets['simulation_list'].count()):
            item = self.widgets['simulation_list'].item(i)
            item.setCheckState(Qt.Checked)

    def _on_btn_sim_all_off(self):
        """Signal for when the user clicks the All on button."""
        for i in range(self.widgets['simulation_list'].count()):
            item = self.widgets['simulation_list'].item(i)
            item.setCheckState(Qt.Unchecked)
