"""TableDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
from PySide2.QtWidgets import QDialogButtonBox, QVBoxLayout

# 3. Aquaveo modules
from xms.tool_core.table_definition import TableDefinition

# 4. Local modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.widgets.table_with_tool_bar import TableWithToolBar


class TableDialog(XmsDlg):
    """A simple dialog with a table view and toolbar (TableWithToolBar which uses QxPandasTableModel)."""
    def __init__(
        self, title: str, table_definition: TableDefinition, data_frame: pd.DataFrame, parent=None, dlg_name: str = ''
    ):
        """Initializes the class.

        Args:
            title (str): Window title.
            table_definition (TableDefinition): Defines the table (column types, fixed row count or not).
            data_frame(pandas.DataFrame): The data as a Pandas DataFrame.
            parent (QObject): The dialog's Qt parent.
            dlg_name (str): Unique name for this dialog. site-packages import path would make sense.
        """
        super().__init__(parent, dlg_name)
        self.setWindowTitle(title)
        self.setLayout(QVBoxLayout())
        self.table_with_tool_bar = TableWithToolBar(parent=parent)
        self.table_with_tool_bar.setup(table_definition, data_frame)
        self.layout().addWidget(self.table_with_tool_bar)

        # Add the OK/Cancel buttons
        self._button_box = QDialogButtonBox(QDialogButtonBox.Cancel | QDialogButtonBox.Ok, self)
        self._button_box.accepted.connect(self.accept)
        self._button_box.rejected.connect(self.reject)
        self.layout().addWidget(self._button_box)

    def get_values(self) -> pd.DataFrame:
        """Returns the values in the table as a Pandas DataFrame."""
        return self.table_with_tool_bar.get_values()
