"""A Qt XY Series dialog."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import csv
from inspect import Traceback
import os
from pathlib import Path
from typing import Type
import webbrowser

# 2. Third party modules
from matplotlib.backends.backend_qt5agg import FigureCanvas, NavigationToolbar2QT as NavigationToolbar
import matplotlib.dates as mdates
from matplotlib.figure import Figure
import pandas as pd
from pandas import DataFrame
from pandas.plotting import register_matplotlib_converters
from PySide2.QtCore import QDir, QSize, Qt
from PySide2.QtWidgets import QDialogButtonBox, QFileDialog, QHeaderView, QSplitter, QWidget

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment

# 4. Local modules
from xms.guipy.dialogs import list_selector_dialog, message_box
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.dialogs.xy_series_editor_ui import Ui_dlg_xy_series_editor
from xms.guipy.models.qx_pandas_table_model import is_datetime_or_timedelta_dtype, QxPandasTableModel
from xms.guipy.settings import SettingsManager
from xms.guipy.time_format import DEFAULT_DATETIME, ISO_DATETIME_FORMAT, string_to_datetime
from xms.guipy.widgets import widget_builder


class XySeriesEditor(XmsDlg):
    """Dialog to display and edit an XY Series.

    From https://doc-snapshots.qt.io/qtforpython/tutorials/datavisualize/plot_datapoints.html
    """
    def __init__(
        self,
        data_frame,
        series_name,
        dialog_title='XY Series Editor',
        icon=None,
        parent=None,
        readonly_x=False,
        can_add_rows=True,
        stair_step=False,
        read_only=False,
        log_x=False,
        log_y=False
    ):
        """Constructor that takes an in-memory DataFrame.

        If None or nan is found in the y values, the table cells are left blank and gaps are put in the plot.

        If 'XY_UNEDITABLE' is found in the y values, the table cells are left blank and made read only and gaps are
        put in the plot.

        Args:
            data_frame (pandas.DataFrame): The DataFrame.
            series_name (str): The name of the series.
            dialog_title (str): The dialog title.
            icon (QIcon): Icon to use for the dialog.
            parent (QWidget): The dialog parent.
            readonly_x (bool): True to make the X column read only.
            can_add_rows (bool): True if user can add rows.
            stair_step (bool): True to make curve stair-step like. See
            read_only: If True, all columns are readonly, rows can't be added, pasting is disabled, and OK/Cancel
                       becomes just OK.
            log_x (bool): True to use log scale on x-axis.
            log_y (bool): True to use log scale on y-axis.

        """
        super().__init__(parent, 'xms.guipy.dialogs.xy_series_editor')
        self.help_url = 'https://www.xmswiki.com/wiki/XY_Series_Editor'

        self.series_name = series_name
        self.dialog_title = dialog_title
        self.readonly_x = readonly_x
        self.can_add_rows = can_add_rows
        self.stair_step = stair_step
        self._read_only = read_only
        self._log_x = log_x
        self._log_y = log_y

        self.model = None
        self.plot_model = None
        self.figure = None  # matplotlib.figure Figure
        self.canvas = None  # matplotlib.backends.backend_qt5agg FigureCanvas
        self._toolbar = None
        self.ax = None  # matplotlib Axes
        self.in_on_spn_rows = False
        self.in_setup = True
        self.start_dir = ''  # Last directory used on Import
        self._import_filter = ''  # Last filter used on Import
        self.icon = icon

        register_matplotlib_converters()  # I think this is required by mdates.date2num call below

        self.ui = Ui_dlg_xy_series_editor()
        self.setup_ui(data_frame)
        self.ui.tbl_xy_data.paste_delimiter = ' |\t'  # Support pasting space or tab delimited text
        for col in data_frame.columns:
            if is_datetime_or_timedelta_dtype(data_frame[col]):
                # don't allow space or you can't paste dates that include times
                self.ui.tbl_xy_data.paste_delimiter = '\t'
        self.setFocus(Qt.OtherFocusReason)  # Set focus to dialog so ESC button is handled
        self.ui.buttonBox.helpRequested.connect(self.help_requested)

    def setup_ui(self, df):
        """Sets up the UI.

        Args:
            df (pandas.DataFrame): Pandas DataFrame
        """
        self.ui.setupUi(self)

        if self.icon:
            self.setWindowIcon(self.icon)
        self.setWindowTitle(self.dialog_title)

        self.setup_model(df)
        self.setup_table()
        self.setup_rows_spin_control()
        self.setup_context_menus()
        self.setup_plot()
        self.add_splitter()
        self._restore_settings()
        self._handle_read_only()

        # Signals
        self.ui.spn_row_count.valueChanged.connect(self.on_spn_rows)
        self.ui.btn_import.clicked.connect(self.on_btn_import)

        self.in_setup = False

    def _exception_hook(self, type: Type, value: Exception, tback: Traceback):
        """
        Called when an exception occurs on the UI thread.

        Qt suppresses exceptions that occur on the UI thread. This method allows dialogs to inspect and handle them
        instead. See https://stackoverflow.com/questions/1015047/logging-all-exceptions-in-a-pyqt4-app

        The default implementation logs all exceptions to the Python debug log, displays a generic error message, and
        forwards the exception to the original system exception hook. That hook just prints the stack trace to stdout,
        which in our case is normally just thrown away.

        Overrides can either compare `type` to the type of exception they want, or use
        `xms.guipy.dialogs.xms_excepthook.exceptions_equal` to compare their arguments too.

        Overrides should probably be structured something like "If I know how to handle this exception then handle it
        and return. Otherwise, call `super()._exception_hook(type, value, tback)` to let the base class deal with it".

        Args:
            type: Type of the exception that was raised, e.g. `AssertionError`.
            value: The exception that was raised, e.g. `AssertionError('Something went wrong')`.
            tback: Traceback for the exception that was raised. Making decisions based on which line an exception was
                thrown on is bound to be brittle, so most overrides should just pass this through to the base class.
        """
        args = ("'PySide2.QtWidgets.QWidgetItem' object has no attribute 'screenChanged'", )
        if type is AttributeError and value.args == args:
            return

        # Some axis scales don't work for all data, e.g. logit only works on data in [0.0, 1.0].
        if type is OverflowError and value.args == ('cannot convert float infinity to integer', ):
            app_name = XmsEnvironment.xms_environ_app_name()
            message = (
                'The chosen axis scale appears to be unsupported for this data.\n\n'
                'Choose another scale, or Cancel to restore to linear.'
            )
            message_box.message_with_ok(self, message=message, app_name=app_name)

            self.ax.set_xscale('linear')
            self.ax.set_yscale('linear')

            # At this point the plot is basically garbage, so start over again.
            self.add_line_series(self.series_name, self.plot_model, True)
            self.canvas.draw()

            return

        super()._exception_hook(type, value, tback)

    def _handle_read_only(self) -> None:
        """Makes the dialog read-only if necessary."""
        if not self._read_only:
            return

        self.ui.spn_row_count.setEnabled(False)
        self.ui.btn_import.setEnabled(False)
        self.ui.buttonBox.setStandardButtons(QDialogButtonBox.Ok | QDialogButtonBox.Help)

    def setup_model(self, df):
        """Sets up the model."""
        self.model = QxPandasTableModel(df)
        _, defaults = self.column_info()
        self.model.set_default_values(defaults)
        self.model.dataChanged.connect(self.on_data_changed)
        self.model.set_show_nan_as_blank(True)

        self.setup_plot_model()

    def setup_plot_model(self):
        """Creates a QxPandasTableModel for the plot.

        Alters the data if necessary, e.g. stair-step.
        """
        # Create a copy of the dataframe for the plot model
        plot_df = self.model.data_frame.copy()
        if len(plot_df) > 0:
            if self.stair_step:
                plot_df = self.make_stair_step(plot_df)

            # Convert all non-numeric entries in the Y column to math.nan (mathplotlib requires this)
            if pd.api.types.is_object_dtype(plot_df.iloc[:, 1]):
                plot_df.iloc[:, 1] = pd.to_numeric(plot_df.iloc[:, 1], errors='coerce')

        self.plot_model = QxPandasTableModel(plot_df)
        _, defaults = self.column_info()
        self.plot_model.set_default_values(defaults)

    def make_stair_step(self, df: pd.DataFrame) -> pd.DataFrame:
        """Make the data stair-step by inserting values.

        See the example documented in the __init__() method docstrings to understand this.
        """
        new_x, new_y = get_step_function(df.iloc[:, 0].to_list(), df.iloc[:, 1].to_list())
        new_df = pd.DataFrame({df.columns[0]: new_x, df.columns[1]: new_y})
        new_df.index += 1
        return new_df

    def add_splitter(self):
        """Adds a QSplitter between the tables so the sizes can be adjusted."""
        # The only way this seems to work right is to parent it to
        # self and then insert it into the layout.
        splitter = QSplitter(self)
        splitter.setOrientation(Qt.Horizontal)
        splitter.addWidget(self.ui.wid_left)
        splitter.addWidget(self.ui.grp_plot)
        # Just use a fixed starting width of 300 for the table for now
        splitter.setSizes([300, 600])
        splitter.setChildrenCollapsible(False)
        splitter.setStyleSheet(
            'QSplitter::handle:horizontal { background-color: lightgrey; }'
            'QSplitter::handle:vertical { background-color: lightgrey; }'
        )
        pos = self.ui.hlay_main.indexOf(self.ui.grp_plot)
        self.ui.hlay_main.insertWidget(pos, splitter)

    def setup_rows_spin_control(self):
        """Sets up the 'Number of rows' spin control."""
        self.ui.spn_row_count.setMinimum(0)
        self.ui.spn_row_count.setMaximum(100000)
        self.ui.spn_row_count.setValue(self.model.rowCount())
        self.ui.spn_row_count.setKeyboardTracking(False)
        self.ui.spn_row_count.setEnabled(self.can_add_rows)

    def add_series(self):
        """Adds the XY line series to the plot."""
        if not self.ax:
            self.ax = self.figure.add_subplot(111)
        self.add_line_series(self.series_name, self.plot_model, True)
        self.canvas.draw()

    def setup_plot(self):
        """Sets up the plot."""
        self.figure = Figure()
        self.figure.set_layout_engine(layout='tight')  # Frames the plots
        self.canvas = FigureCanvas(self.figure)
        self.canvas.setMinimumWidth(100)  # So user can't resize it to nothing
        self._toolbar = NavigationToolbar(self.canvas, self)
        # Remove the "Configure subplots" button. We aren't really sure what this does.
        for x in self._toolbar.actions():
            if x.text() == 'Subplots':
                self._toolbar.removeAction(x)

        self.ui.vlay_grp_box.addWidget(self._toolbar)
        self.ui.vlay_grp_box.addWidget(self.canvas)
        self.ui.txt_plots.setVisible(False)
        self.ui.cbx_selected_plot.setVisible(False)
        num_columns = self.plot_model.data_frame.shape[1]
        plot_yy = num_columns > 2
        if plot_yy:
            self.ui.txt_plots.width = self.ui.btn_import.width
            self.ui.txt_plots.height = self.ui.btn_import.height
            self.ui.txt_plots.setVisible(True)
            self.ui.cbx_selected_plot.width = self.ui.btn_import.width
            self.ui.cbx_selected_plot.height = self.ui.btn_import.height
            self.ui.cbx_selected_plot.clear()
            self.ui.cbx_selected_plot.setVisible(True)
            for col in range(1, num_columns):
                self.ui.cbx_selected_plot.addItem(self.plot_model.data_frame.columns[col])
            self.ui.cbx_selected_plot.currentIndexChanged.connect(self.on_plot_item_click)
        self.add_series()

    def setup_table(self):
        """Sets up the table."""
        self.ui.tbl_xy_data.size_to_contents = True
        widget_builder.style_table_view(self.ui.tbl_xy_data)
        self.ui.tbl_xy_data.setModel(self.model)
        if self.readonly_x:
            self.model.set_read_only_columns({0})
        if self._read_only:
            self.model.set_read_only_columns({column_idx for column_idx in range(self.model.columnCount())})

        for row in range(self.model.rowCount()):
            s = str(self.model.index(row, 1).data())
            if 'XY_UNEDITABLE' in s:
                index = self.model.createIndex(row, 1)
                new_str = s.replace('XY_UNEDITABLE', '')
                self.model.setData(index, new_str)
                self.model.read_only_cells.add((row, 1))

        # QTableView Headers
        # resize_mode = (QHeaderView.ResizeToContents | QHeaderView.Interactive)
        resize_mode = QHeaderView.Interactive
        horizontal_header = self.ui.tbl_xy_data.horizontalHeader()
        vertical_header = self.ui.tbl_xy_data.verticalHeader()
        horizontal_header.setSectionResizeMode(resize_mode)
        vertical_header.setSectionResizeMode(resize_mode)
        self.ui.tbl_xy_data.setColumnWidth(1, 1)  # Size column 1 to 1 to stop it from remembering an old, larger size
        self.ui.tbl_xy_data.resizeColumnsToContents()
        horizontal_header.setStretchLastSection(True)
        self.ui.tbl_xy_data.adjustSize()  # Seems to be needed after we import data

    def setup_context_menus(self):
        """Sets up the context menus."""
        self.ui.tbl_xy_data.verticalHeader().setContextMenuPolicy(Qt.CustomContextMenu)
        self.ui.tbl_xy_data.verticalHeader().customContextMenuRequested.connect(self.on_index_column_click)
        self.ui.tbl_xy_data.setContextMenuPolicy(Qt.CustomContextMenu)
        self.ui.tbl_xy_data.customContextMenuRequested.connect(self.on_right_click)

    def on_right_click(self, point):
        """Slot called when user right-clicks in the table.

        Args:
            point(QPoint): The point clicked.
        """
        # row = self.ui.table_view.logicalIndexAt(point)
        if self._read_only:
            menu_list = [['', 'Copy', self.ui.tbl_xy_data.on_copy]]
        else:
            menu_list = [['', 'Copy', self.ui.tbl_xy_data.on_copy], ['', 'Paste', self.ui.tbl_xy_data.on_paste]]
        menu = widget_builder.setup_context_menu(self, menu_list)
        menu.popup(self.ui.tbl_xy_data.viewport().mapToGlobal(point))

    def on_index_column_click(self, point):
        """Called on right-click on the index column (vertical header).

        Args:
            point (QPoint): The point clicked
        """
        row = self.ui.tbl_xy_data.verticalHeader().logicalIndexAt(point)
        self.ui.tbl_xy_data.selectRow(row)
        menu_list = [
            ['icons/row-insert.svg', 'Insert', self.on_insert_rows],
            ['icons/row-delete.svg', 'Delete', self.on_delete_rows], ['', 'Copy', self.ui.tbl_xy_data.on_copy]
        ]
        if not self._read_only:
            menu_list.append(['', 'Paste', self.ui.tbl_xy_data.on_paste])
        menu = widget_builder.setup_context_menu(self, menu_list)
        menu.popup(self.ui.tbl_xy_data.viewport().mapToGlobal(point))

    def on_btn_import(self):
        """Import a .xys file."""
        # Show Open File dialog to user so they can pick a file
        filters = ['XY Series Files (*.xys)', 'CSV (Comma delimited) Files (*.csv)', 'All files (*.*)']
        filters_string = ';;'.join(filters)

        if not self.start_dir:
            self.start_dir = QDir.homePath()
        if not self._import_filter:
            self._import_filter = filters[0]

        filename, selected_filter = QFileDialog.getOpenFileName(
            parent=self,
            caption='Open',
            dir=str(self.start_dir),
            filter=filters_string,
            selectedFilter=self._import_filter,
        )
        if not filename:
            return
        self.start_dir = os.path.dirname(filename)
        self._import_filter = selected_filter

        is_xy = False
        try:
            filepath = Path(filename)
            columns = self.model.data_frame.columns.to_list()

            if selected_filter == 'XY Series Files (*.xys)' or filepath.suffix == '.xys':
                is_xy = True
                rv = _import_xys_file(filepath, columns, self)
                if rv is None:
                    return
                df, self.series_name = rv[0], rv[1]
            elif selected_filter == 'CSV (Comma delimited) Files (*.csv)' or filepath.suffix == '.csv':
                df = import_csv_file(filepath, columns)
            else:
                raise RuntimeError('Unknown file format.')

            # Setup the table and the plot
            self.stair_step = False  # If this was on, turn it off.
            self.setup_model(df)
            self.setup_table()
            self.setup_rows_spin_control()
            self.add_series()

        except Exception:
            if is_xy:  # If an XY series file, direct user to wiki even though we might not be SMS. Format is same.
                msg = (
                    'Unexpected error encountered while importing XY series file. Please ensure format is correct. '
                    'See https://xmswiki.com/wiki/SMS:XY_Series_Files for more details.'
                )
            else:
                msg = 'Unexpected error encountered while CSV file. Please ensure format is correct.'
            message_with_ok(parent=self, message=msg, app_name='', icon='Critical', win_icon=self.icon)

    def on_spn_rows(self):
        """Called when rows spin control is changed."""
        if self.in_on_spn_rows:  # Don't come in here recursively
            return
        self.in_on_spn_rows = True

        rows = self.ui.spn_row_count.value()
        rows_old = self.model.rowCount()
        diff = abs(rows - rows_old)
        if rows > rows_old:
            self.model.insertRows(rows_old, diff)
        elif rows_old > rows:
            self.model.removeRows(rows_old - diff, diff)

        self.in_on_spn_rows = False

    def on_insert_rows(self):
        """Inserts rows into the table."""
        selected_list = self.ui.tbl_xy_data.selectedIndexes()
        if not selected_list:
            return

        minrow = selected_list[0].row()
        self.model.insertRows(minrow, 1)
        self.ui.spn_row_count.setValue(self.model.rowCount())

    def on_delete_rows(self):
        """Deletes rows from the table."""
        selected_list = self.ui.tbl_xy_data.selectedIndexes()
        if not selected_list:
            return

        minrow = selected_list[0].row()
        self.model.removeRows(minrow, 1)
        self.ui.spn_row_count.setValue(self.model.rowCount())

    def column_info(self):
        """Returns the column names and default values.

        Returns:
            (tuple): tuple containing:

                columns (list[str]): List of column names.

                defaults (dict{str:values}): Dictionary of column default values.

        """
        columns = list(self.model.data_frame.columns)
        defaults = {}
        for column in columns:
            if is_datetime_or_timedelta_dtype(self.model.data_frame[column]):
                defaults[column] = widget_builder.datetime_from_string_using_qt('2001-01-01')
            else:
                defaults[column] = 0.0
        return columns, defaults

    def on_data_changed(self, top_left_index, bottom_right_index):
        """Called when the data in the table view has changed.

        Args:
            top_left_index (QModelIndex): Top left index.
            bottom_right_index (QModelIndex): Bottom right index.
        """
        if self.in_setup:
            return

        del top_left_index, bottom_right_index  # Unused parameters
        if self.ui.tbl_xy_data.pasting:
            return

        self.setup_plot_model()
        self.add_series()

        row_count = self.model.rowCount()
        self.ui.spn_row_count.setValue(row_count)

    def sizeHint(self):  # noqa: N802
        """Overridden method to help the dialog have a good minimum size.

        Returns:
            (QSize): Size to use for the initial dialog size.
        """
        return QSize(800, 500)

    def add_line_series(self, name, model, show_axis_titles):
        """Adds an XY line series to the plot.

        Args:
            name (str): The series name.
            model(QxPandasTableModel): The model.
            show_axis_titles (bool): If True, axis titles displayed

        """
        self.ax.clear()
        self.ax.set_title(name)
        self.ax.grid(True)

        if model.rowCount() == 0:
            return

        # Add data to plot
        x_column = model.data_frame.iloc[:, 0]
        sel_plot = 0
        if self.ui.cbx_selected_plot.isVisible():
            sel_plot = self.ui.cbx_selected_plot.currentIndex()
        x_is_date_times = is_datetime_or_timedelta_dtype(x_column)
        if x_is_date_times:
            dates = mdates.date2num(x_column)
            self.ax.plot(dates, model.data_frame.iloc[:, sel_plot + 1], label=model.data_frame.columns[sel_plot + 1])
        else:
            self.ax.plot(x_column, model.data_frame.iloc[:, sel_plot + 1], label=model.data_frame.columns[sel_plot + 1])

        if self._log_x and not x_is_date_times:
            self.ax.set_xscale('log', nonpositive='mask')
        if self._log_y:
            self.ax.set_yscale('log', nonpositive='mask')

        # X-axis format
        if x_is_date_times:
            # self.ax.format_xdata = mdates.DateFormatter(ISO_DATETIME_FORMAT)  # Seems to be overridden by next line
            formatter = mdates.DateFormatter(ISO_DATETIME_FORMAT)  # Seems to be overridden by next line
            self.ax.xaxis.set_major_formatter(formatter)
            self.figure.autofmt_xdate()
        else:
            # self.ax.xaxis.set_major_formatter(mpl.ticker.StrMethodFormatter('{x}'))
            pass

        # Axis titles
        if show_axis_titles:
            self.ax.set_xlabel(model.data_frame.columns[0])
        if show_axis_titles:
            self.ax.set_ylabel(model.data_frame.columns[sel_plot + 1])

    def on_plot_item_click(self):
        """Called when user toggles the display of one of the plots."""
        self.add_series()

    def _save_settings(self):
        """Saves some things to the registry."""
        settings = SettingsManager()
        settings.save_setting(self._dlg_name, 'import_directory', self.start_dir)
        settings.save_setting(self._dlg_name, 'import_filter', self._import_filter)

    def _restore_settings(self):
        """Restores some settings."""
        settings = SettingsManager()
        self.start_dir = settings.get_setting(self._dlg_name, 'import_directory', '')
        self._import_filter = settings.get_setting(self._dlg_name, 'import_filter', '')
        # If no start_dir, try to use the XMS project path (if the project has been saved)
        if not self.start_dir or not Path(self.start_dir).is_dir():
            project_file = XmsEnvironment.xms_environ_project_path()
            if project_file:
                self.start_dir = Path(XmsEnvironment.xms_environ_project_path()).parent

    def accept(self):
        """Called when OK button is clicked."""
        self._save_settings()
        super().accept()

    def reject(self):
        """Called when the Cancel button is clicked."""
        self._save_settings()
        super().reject()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)


def get_step_function(x_values, y_values) -> tuple[list[any], list[any]]:
    r"""Returns new values making it a step function.

    THIS IS COPIED FROM xmscoverage.xy.xy_util BECAUSE xmscoverage DEPENDS ON xmxguipy.

    ::

              Before                          After
                  *                             *---*
                /   \                           |   |
      y       *       *                     *---*   *
      |
      *--x

    Args:
        x_values: The x values.
        y_values: The y values

    Return:
        (tuple(list[any], list[any])): New values as a step function.
    """
    if not x_values or len(x_values) < 2:
        return x_values, y_values

    new_x = []
    new_y = []
    for i, (x, y) in enumerate(zip(x_values, y_values)):
        new_x.append(x)
        new_y.append(y)
        if i < len(x_values) - 1:
            new_x.append(x_values[i + 1])
            new_y.append(y)
    return new_x, new_y


def _pick_series_from_list(series_names: list[str], win_cont: QWidget) -> int | None:
    """Show dialog letting user pick which series to import from a list and return index of choice, or None on cancel.

    Args:
        series_names: List of xy series names found in file
        win_cont: Window parent

    Returns:
        See description
    """
    title = 'Xy Series in File'
    heading = 'Choose which XY Series to import'
    picks = list_selector_dialog.run(title, series_names, heading, False, True, win_cont)
    if not picks:
        return None
    else:
        return series_names.index(picks[0])


def _import_xys_file(filepath: Path,
                     columns: list[str],
                     win_cont: QWidget = None,
                     series_idx: int | None = None) -> tuple[DataFrame, str] | None:
    """Imports a .xys file and returns a dataframe.

    Args:
        filepath (Path): The filepath.
        columns (list[str]): List of column names
        series_idx: If given, the index of the series in the file. Useful when testing if file has multiple series.

    Returns:
        (tuple(pandas.DataFrame, str)): The dataframe and series name.
    """
    # Find all the xy series in the file
    with filepath.open('r') as file:
        xy_series = []  # List of tuples of: series name, entire XYS (or XY3) line, position of file at end of line
        line = file.readline()  # Can't use "for line in file" because we call tell()
        while line:
            if line.startswith('XYS'):
                words = line.split()
                xy_series.append((words[3].strip('"'), line, file.tell()))
            elif line.startswith('XY3'):
                words = line.split()
                xy_series.append((words[10].strip('"'), line, file.tell()))
            line = file.readline()  # Can't use "for line in file" because we call tell()
        if not xy_series:
            raise RuntimeError('No XYS or XY3 cards found in file.')

        # Set which series, if there are multiple
        if len(xy_series) == 1:
            series_idx = 0
        elif series_idx is None:
            series_idx = _pick_series_from_list([name for name, line, pos in xy_series], win_cont)
            if series_idx is None:
                return None

        # Read the series using Pandas.read_csv
        file.seek(xy_series[series_idx][2])
        words = xy_series[series_idx][1].split()
        xy_series_type = words[0].upper()
        name = xy_series[series_idx][0]
        n = int(words[2])
        if xy_series_type == 'XYS':
            df = pd.read_csv(
                file,
                # header=None,
                names=columns,
                sep=' ',
                float_precision='high',
                index_col=False,
                nrows=n
            )
        else:  # xy_series_type == 'XY3':  # WMS and GSSHA still use XY3
            # XY3 id n x1 incx biasx dx dy rep begc name
            x1, incx, biasx = float(words[3]), float(words[4]), float(words[5])
            df = pd.read_csv(
                file,
                # header=None,
                names=[columns[1]],
                float_precision='high',
                index_col=False,
                nrows=n
            )

            # Calculate x values
            x = [x1] * n
            for i in range(1, n):
                x[i] = x[i - 1] + incx
                incx *= (1.0 + biasx)
            df.insert(0, columns[0], x)

        if df is not None:
            df.index += 1
            _set_dtype_on_first_column(df)
    return df, name


def import_csv_file(filepath: Path, columns: list[str]) -> DataFrame:
    """Imports a .xys file and returns a dataframe.

    Args:
        filepath (Path): The filepath.
        columns (list[str]): List of column names. If empty, headers from file (if any) are used.

    Returns:
        pandas.DataFrame: The dataframe and series name.
    """
    if _has_headers(filepath):
        df = pd.read_csv(filepath, header=0, float_precision='high', index_col=False)
    else:
        df = pd.read_csv(filepath, header=None, float_precision='high', index_col=False)
    if columns:
        new_names = {df.columns[i]: columns[i] for i in range(2)}
        df.rename(columns=new_names, inplace=True)
    df.index += 1
    _set_dtype_on_first_column(df)
    return df


def _has_headers(csv_file: Path | str) -> bool:
    """Returns True if the .csv file contains column headers.

    Args:
        csv_file:

    Returns:
        See description.
    """
    with open(csv_file, 'r') as f:
        reader = csv.reader(f)
        first_row = next(reader)  # Read the first row
        # If all values are either floats or dates assume they are not headers
        has_headers = not all(_is_float(x) or string_to_datetime(x) != DEFAULT_DATETIME for x in first_row)
    return has_headers


def _set_dtype_on_first_column(df: DataFrame):
    """Sets the dtype on the first column to be a datetime if it is not a float.

    Args:
        df (pandas.DataFrame): The DataFrame.
    """
    # If X column is not a float, assume it's a date/time
    if not _is_float(df.iloc[0, 0]):
        # df[df.columns[0]] = df[df.columns[0]].astype(dtype='datetime64')
        df[df.columns[0]] = pd.to_datetime(df[df.columns[0]], format='mixed')


def _is_float(value):
    try:
        float(value)
    except ValueError:
        return False
    return True
