"""Widget enabler/disabler for param objects."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import param

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.param import param_util


class ParamEnabler:
    """Param based data class."""
    def __init__(self, param_cls, depends, enablers):
        """Initializes the data class.

        Args:
            param_cls (param.Parameterized): class with param objects
            depends (dict): Dict of dependencies between param objects
            enablers (list[str]): list of param object names

        """
        super().__init__()
        self._param_cls = param_cls
        self._enablers = enablers
        self._depends = depends
        self.do_enabling(force_disable=False)

    def do_enabling(self, **kwargs):
        """One method to handle setting param object precedence for all dependencies.

        Args:
            force_disable (bool): flag to force disabling of items in self.depends and exit
        """
        if 'force_disable' not in kwargs:
            raise RuntimeError
        force_disable = kwargs['force_disable']
        if force_disable:
            self._disable_all_param()
            return

        self._restore_all_precedence()
        self._disable_all_param_in_depends()
        enable_list = []
        for pname in self._enablers:
            value = getattr(self._param_cls, pname)
            # if this is a param.Boolean we want the param name used to enable/disable
            if type(value) is bool and value is True:
                value = pname
                enable_list.append(value)
            else:
                enable_list.append(value)
        self._enable_param(options=enable_list)

        names, params = param_util.names_and_params_from_class(self._param_cls)
        for i in range(len(params)):
            subclass_force_disable = force_disable or params[i].precedence < 0
            self._recurse_param_base(params[i], names[i], 'do_enabling', force_disable=subclass_force_disable)

    def _recurse_param_base(self, param_obj, name, method, **kwargs):
        """Get and execute a param object's method dynamically.

        Args:
            param_obj: : param class instance
            name: Component name
            method (str): Method to get from class
            **kwargs: keyword arguements
        """
        if type(param_obj) is param.ClassSelector:
            val = getattr(self._param_cls, name)
            if val is not None and hasattr(val, 'enabler'):
                getattr(val.enabler, method)(**kwargs)

    def _disable_all_param(self, **kwargs):
        """Disable all param objects."""
        names, params = param_util.names_and_params_from_class(self._param_cls)
        for i in range(len(params)):
            params[i].precedence = -1
            self._recurse_param_base(params[i], names[i], '_disable_all_param')

    def _restore_all_precedence(self, **kwargs):
        """Restore the precedence for the param objects to original state."""
        class_objs_dict = self._param_cls.param.objects(instance=False)
        names, params = param_util.names_and_params_from_class(self._param_cls)
        for i in range(len(names)):
            params[i].precedence = class_objs_dict[names[i]].precedence
            self._recurse_param_base(params[i], names[i], '_restore_all_precedence')

    def _disable_all_param_in_depends(self):
        """Disable all param objects in the depends dictionary of the class."""
        p = self._param_cls.param
        for param_names_list in self._depends.values():
            for param_name in param_names_list:
                obj = getattr(p, param_name)
                # we are disabling a param class containing param objects, so we want to force everything
                # to be disabled
                obj.precedence = -1
                self._recurse_param_base(obj, param_name, 'do_enabling', force_disable=True)

    def _enable_param(self, **kwargs):
        """Enable param objects that are listed under option in cls.depends.

        Args:
            options (list[str]): option strings
        """
        if 'options' not in kwargs.keys():
            raise RuntimeError
        class_obj_dict = self._param_cls.param.objects(instance=False)
        options = kwargs['options']
        for opt in options:
            if opt in self._depends:
                lst = self._depends[opt]
                for val in lst:
                    obj = getattr(self._param_cls.param, val)
                    obj.precedence = class_obj_dict[val].precedence
                    self._recurse_param_base(obj, val, '_enable_param', **kwargs)
