"""param helper utility methods for xmsguipy."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
import param

# 3. Aquaveo modules

# 4. Local modules


def fill_param_class_from_dict(param_cls, val_dict):
    """Fills the values of members of a param class.

    Args:
       param_cls (param.Parameterized): class with param objects
       val_dict (dict): dictionary with names from param_class_to_names_values and values
    """
    cls_name = param_cls.__class__.__name__
    pnames, params = names_and_params_from_class(param_cls)
    cls_list = [(cls_name, pnames, params, param_cls)]
    for item in cls_list:
        cls_name = item[0]
        par_cls = item[3]
        for i in range(len(item[1])):
            name = item[1][i]
            par = item[2][i]
            val = getattr(par_cls, name)
            if issubclass(type(val), param.Parameterized):
                new_name = f'{cls_name}.{val.__class__.__name__}'
                new_names, new_params = names_and_params_from_class(val)
                cls_list.append((new_name, new_names, new_params, val))  # noqa: B038 editing a loop's mutable iterable
            else:
                name_id = f'{cls_name}.{name}'
                if name_id in val_dict and val_dict[name_id] is not None:
                    val = val_dict[name_id]
                    if val is not None:
                        if type(val) is pd.DataFrame:
                            df = getattr(par_cls, name)
                            df_col = df.columns
                            val.rename(columns={val.columns[0]: df_col[0], val.columns[1]: df_col[1]}, inplace=True)
                        elif type(par) is param.Integer:
                            val = int(val)
                        elif type(par) is param.Boolean:
                            val = bool(val)
                        setattr(par_cls, name, val)


def param_class_to_names_values(param_cls):
    """Returns a list of param names and values.

    Args:
       param_cls (param.Parameterized): class with param objects

    Return:
        names(list[str]), values(list[values])
    """
    cls_name = param_cls.__class__.__name__
    pnames, params = names_and_params_from_class(param_cls)
    cls_list = [(cls_name, pnames, params, param_cls)]

    names = []
    values = []
    for item in cls_list:
        cls_name = item[0]
        par_cls = item[3]
        for i in range(len(item[1])):
            name = item[1][i]
            param_obj = item[2][i]
            val = getattr(par_cls, name)
            if type(param_obj) is param.Action:
                continue

            if issubclass(type(val), param.Parameterized):
                new_name = f'{cls_name}.{val.__class__.__name__}'
                new_names, new_params = names_and_params_from_class(val)
                cls_list.append((new_name, new_names, new_params, val))  # noqa: B038 editing a loop's mutable iterable
            else:
                names.append(f'{cls_name}.{name}')
                values.append(val)
    return names, values


def names_and_params_from_class(param_cls):
    """Returns a list of param names and objects.

    Args:
       param_cls (param.Parameterized): class with param objects

    Return:
        names(list[str]), params(list[param objects])
    """
    names = []
    params = []
    p = param_cls.param
    lst = p.values().items()
    for item in lst:
        if item[0] in ['name']:
            continue
        obj = getattr(p, item[0])
        if obj is not None:
            names.append(item[0])
            params.append(obj)
    return names, params


def declared_precedence_dict(param_cls):
    """Returns a dict where [param_name] = param_precedence.

    Args:
       param_cls (param.Parameterized): class with param objects

    Return:
        precedence(dict[str] = float)
    """
    class_objs = param_cls.param.objects(instance=False)
    precedence = dict()
    for key, val in class_objs.items():
        if key == 'name':
            continue
        precedence[key] = val.precedence
    return precedence


def param_cls_to_json_list(cls):
    """List of data the will write to json.

    Args:
       cls (param.Parameterized): class with param objects

    Return:
        (list[dict()])
    """
    pnames, params = names_and_params_from_class(cls)
    class_obj_dict = cls.param.objects(instance=False)
    plist = []
    for i in range(len(pnames)):
        plist.append((class_obj_dict[pnames[i]].precedence, pnames[i], params[i]))
    plist.sort()

    output = []
    for i in range(len(plist)):
        pname = plist[i][1]
        if pname == 'name':
            continue
        obj = plist[i][2]  # par
        if hasattr(obj, 'precedence') and obj.precedence < 0:
            continue
        d = dict()
        val = getattr(cls, pname)
        if issubclass(type(val), param.Parameterized):
            d[pname] = param_cls_to_json_list(val)
        else:
            d[pname] = val
        output.append(d)
    return output


def json_list_to_param_cls(plist, cls):
    """Changes param values from a json list.

    Args:
       plist (list[dict]): json list of param data see param_cls_to_json_list
       cls (param.Parameterized): class with param objects
    """
    for d in plist:
        for key, val in d.items():
            if not hasattr(cls, key):
                continue
            obj = getattr(cls, key)
            obj_type = type(obj)
            if issubclass(obj_type, param.Parameterized):
                json_list_to_param_cls(val, obj)
            else:
                setattr(cls, key, obj_type(val))


def orjson_dict_to_param_cls(pdict, cls, df_handler):
    """Create a param object from a deserialized json dict.

    Args:
        pdict (dict): json dict of param data
        cls (param.Parameterized): class with param objects
        df_handler (method): method to handle DataFrame
    """
    for key, val in pdict.items():
        if not hasattr(cls, key):  # pragma: no cover
            continue  # pragma: no cover
        obj = getattr(cls, key)
        obj_type = type(obj)
        if issubclass(obj_type, param.Parameterized):
            orjson_dict_to_param_cls(val, obj, df_handler)
        else:
            if type(obj) is pd.DataFrame:
                if df_handler is not None:
                    df = df_handler(val)
                    df.columns = obj.columns
                    setattr(cls, key, df)
            else:
                setattr(cls, key, obj_type(val))


def param_cls_to_orjson_dict(cls, df_handler, skip_negative_precedence=True):
    """Convert a param class to a dict suitable for serialization with orjson.

    Args:
        cls (param.Parameterized): class with param objects
        df_handler (method): method to handle DataFrame
        skip_negative_precedence (bool): skip parameters with negative precedence

    Return:
        (dict): param name, value pairs
    """
    pnames, params = names_and_params_from_class(cls)
    class_obj_dict = cls.param.objects(instance=False)
    plist = []
    for i in range(len(pnames)):
        plist.append((class_obj_dict[pnames[i]].precedence, pnames[i], params[i]))
    plist.sort()

    output = {}
    for i in range(len(plist)):
        pname = plist[i][1]
        obj = plist[i][2]  # par
        if type(obj) is param.Action:
            continue
        if skip_negative_precedence and hasattr(obj, 'precedence') and obj.precedence < 0:
            continue
        val = getattr(cls, pname)
        if issubclass(type(val), param.Parameterized):
            output[pname] = param_cls_to_orjson_dict(val, df_handler, skip_negative_precedence)
        else:
            if type(obj) is param.DataFrame:
                if df_handler is not None:
                    df_out = df_handler(val)
                    output[pname] = df_out
            else:
                output[pname] = val
    return output
