"""Classes and methods for formatting dates and times consistently with XMS."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime, timedelta
import locale

# 2. Third party modules
from dateutil import parser
import orjson
from PySide2.QtCore import QDate, QDateTime, QLocale, QTime

# 3. Aquaveo modules

# 4. Local modules

ISO_DATETIME_FORMAT = '%Y-%m-%d %H:%M:%S'  # One we like the best, microseconds are trimmed

KEY_ABS_DATE_FORMAT = 'DisplayFormatAbsDate'
KEY_ABS_TIME_FORMAT = 'DisplayFormatAbsTime'
KEY_REL_FORMAT = 'DisplayFormatRel'
KEY_DISPLAY_REF_TYPE = 'DisplayRefType'

# Written by SMS to registry
# KEY_AVAILABLE_TIMES = 'AvailableTimes'
# KEY_TIME_STEP_ROUNDING = 'TimeStepRounding'
# KEY_REL_FORMAT_PREC = 'RelativeFormatPrecision'

DISPLAY_ABSOLUTE_TIMES = 0
DISPLAY_RELATIVE_TIMES = 1

FOR_ABS_DATE_REGIONAL = 0
FOR_ABS_DATE_mm_dd_yy = 1
FOR_ABS_DATE_dd_mm_yy = 2
FOR_ABS_DATE_mm_dd_yyyy = 3
FOR_ABS_DATE_dd_mm_yyyy = 4
FOR_ABS_DATE_dd__mon__yy = 5
FOR_ABS_DATE_dd__mon__yyyy = 6
FOR_ABS_DATE_mon_day_comma_year = 7
ABS_DATE_FORMAT_SPECIFIERS = {
    FOR_ABS_DATE_REGIONAL: '',
    FOR_ABS_DATE_mm_dd_yy: '%m/%d/%y',
    FOR_ABS_DATE_dd_mm_yy: '%d/%m/%y',
    FOR_ABS_DATE_mm_dd_yyyy: '%m/%d/%Y',
    FOR_ABS_DATE_dd_mm_yyyy: '%d/%m/%Y',
    FOR_ABS_DATE_dd__mon__yy: '%d-%b-%y',
    FOR_ABS_DATE_dd__mon__yyyy: '%d-%b-%Y',
    FOR_ABS_DATE_mon_day_comma_year: '%b %d, %Y',
}

FOR_ABS_TIME_REGIONAL = 0
FOR_ABS_TIME_12hh_c_mm_ampm = 1
FOR_ABS_TIME_24hh_c_mm = 2
FOR_ABS_TIME_12hh_c_mm_ss_ampm = 3
FOR_ABS_TIME_24hh_c_mm_c_ss = 4
ABS_TIME_FORMAT_SPECIFIERS = {
    FOR_ABS_TIME_REGIONAL: '',
    FOR_ABS_TIME_12hh_c_mm_ampm: '%#I:%M %p',
    FOR_ABS_TIME_24hh_c_mm: '%#H:%M',
    FOR_ABS_TIME_12hh_c_mm_ss_ampm: '%#I:%M:%S %p',
    FOR_ABS_TIME_24hh_c_mm_c_ss: '%#H:%M:%S',
}

FOR_REL_days_s_hh_c_mm_c_ss = 0
FOR_REL_hours_c_mm_c_ss = 1
FOR_REL_days_s_hh_c_mm = 2
FOR_REL_hours_c_mm = 3
# The following are a single unit with decimal if necessary
FOR_REL_days_decimal = 4
FOR_REL_hours_decimal = 5
FOR_REL_min_decimal = 6
FOR_REL_seconds = 7
FOR_REL_years_decimal = 8

# Mappings of Qt date/time format specifiers to standard strftime specifiers. Note that Qt does not support all the
# format specifiers that strftime does. There are equivalents for most of the specifiers that can be set in the user
# preferences of XMS. If there is not an exact equivalent, the differences are noted. Note that some specifiers are
# Windows-specific. Also note that there is no mapping for sub-second units because Qt represents them as milliseconds
# only, while strftime uses microseconds only. For all these reasons, use with caution. These mappings and the
# conversion functions below should not be used for parsing datetimes, just for setting time formatting in GUI widgets.
QT_TO_STRFTIME_DATES = {  # Keys for each unit are in descending length order so we can replace specifiers in a loop.
    'dddd': '%A',  # locale full weekday name
    'ddd': '%a',   # locale abbreviated weekday name
    'dd': '%d',    # 0-padded day of month
    'd': '%#d',    # day of month without leading 0 (Windows-specific)
    'MMMM': '%B',  # locale full month name
    'MMM': '%b',   # locale abbreviated month name
    'MM': '%m',    # 0-padded month
    'M': '%#m',    # month without leading 0 (Windows-specific)
    'yyyy': '%Y',  # 4 digit year
    'yy': '%y',    # 2 digit year
}
QT_TO_STRFTIME_TIMES = {  # Keys for each unit are in descending length order so we can replace specifiers in a loop.
    # Times
    'HH': '%H',    # 0-padded 24-hour clock
    'H': '%#H',    # 24-hour clock without leading 0 (Windows-specific)
    # Note that Qt will use a 24-hour clock for the next two hour specifiers if an AM/PM specifier is not included, but
    # strftime has no equivalent.
    'hh': '%I',    # 0-padded 12-hour clock
    'h': '%#I',    # 12-hour clock without leading 0 (Windows-specific)
    'mm': '%M',    # 0-padded minute
    'm': '%#M',    # minute without leading 0 (Windows-specific)
    'ss': '%S',    # 0-padded second
    's': '%#S',    # second without leading 0 (Windows-specific)
    # Note that Qt has several options for AM/PM (e.g. lowercase), but strftime only supports formatting in the current
    # locale's equivalent of AM/PM.
    'AP': '%p',    # uppercase AM/PM in Qt, locale equivalent of AM/PM in strftime
    'A': '%p',     # uppercase AM in Qt, locale equivalent of AM/PM in strftime
    'ap': '%p',    # lowercase am/pm in Qt, locale equivalent of AM/PM in strftime
    'a': '%p',     # lowercase am in Qt, locale equivalent of AM/PM in strftime
}

DEFAULT_DATETIME = datetime(1950, 1, 1)


def convert_strftime_specifiers_to_qt(specifier, date):
    """Convert strftime format specifiers to Qt equivalents because Qt is dumb and doesn't use standard specifiers.

    Args:
        specifier (str): The date or time portion of datetime format specifier string
        date (bool): True if the date portion of the datetime, False for the time portion

    Returns:
        (str): Qt version of the standard strftime specifiers
    """
    # First check for default locale format
    specifier = specifier.strip()  # Empty string means use locale default
    if date and (not specifier or specifier == '%x'):
        return QLocale.system().dateFormat(QLocale.ShortFormat)
    elif not date and (not specifier or specifier == '%X'):
        return QLocale.system().timeFormat(QLocale.LongFormat)
    # Convert custom format specifiers
    specifier_map = QT_TO_STRFTIME_DATES if date else QT_TO_STRFTIME_TIMES
    for qt, std in specifier_map.items():
        specifier = specifier.replace(std, qt)
    return specifier


def coledatetime_to_datetime(coledatetime):
    """Convert an MFC COleDateTime float to a Python datetime.

    Args:
        coledatetime (float): The offset in days from the COleDateTime epoch (midnight on Dec 30, 1989)

    Returns:
        (datetime): A Python datetime representing coledatetime
    """
    base_dt = datetime(1899, 12, 30)
    day_offset = timedelta(days=coledatetime)
    return base_dt + day_offset


def datetime_to_qdatetime(dt_literal):
    """Convert a Python datetime object to a QDateTime.

    Args:
        dt_literal (datetime): Datetime to convert

    Returns:
        (QDateTime): See description
    """
    return QDateTime(
        QDate(dt_literal.year, dt_literal.month, dt_literal.day),
        QTime(dt_literal.hour, dt_literal.minute, dt_literal.second)
    )


def qdatetime_to_datetime(qdatetime):
    """Convert a QDateTime to a Python datetime object.

    Args:
        qdatetime (QDateTime): QDateTime to convert

    Returns:
        (datetime): See description
    """
    return datetime(
        year=qdatetime.date().year(),
        month=qdatetime.date().month(),
        day=qdatetime.date().day(),
        hour=qdatetime.time().hour(),
        minute=qdatetime.time().minute(),
        second=qdatetime.time().second()
    )


def string_to_datetime(s: str, default: datetime | None = DEFAULT_DATETIME):
    """Parse a string into a datetime.

    Args:
        s (str): The string to parse.
        default (datetime | None): The default time to return if the string couldn't be parsed.

    Returns:
        (str) The parsed datetime.
    """
    # Try to parse using current locale
    try:
        qreftime = QDateTime.fromString(s)
        if qreftime.isValid():
            return qdatetime_to_datetime(qreftime)
    except Exception:
        pass

    # Try to parse using ISO format
    try:
        return datetime.strptime(s, ISO_DATETIME_FORMAT)
    except Exception:
        pass

    # Try using python-dateutil library
    try:
        return parser.parse(s)
    except Exception:
        pass

    # Give up and use a default value
    return default


def datetime_to_string(date_time):
    """Convert anything resembling a date and time to an ISO datetime string.

    Args:
        date_time (Union[datetime, QDateTime, str]): The datetime to convert. Can be a datetime, QDateTime,
            a locale-specific datetime string, or an ISO standard datetime string. The last one just gives you back the
            same string, so you can just convert everything without having to figure out whether it's safe or not.

    Returns:
        (str) The ISO standard representation of date_time.
    """
    if isinstance(date_time, str):
        date_time = string_to_datetime(date_time)

    if isinstance(date_time, QDateTime):
        date_time = qdatetime_to_datetime(date_time)

    return date_time.strftime(ISO_DATETIME_FORMAT)


class XmsTimeFormatter:
    """Format date/time strings using XMS time settings."""
    def __init__(self, json):
        """Initialize manager based on the XMS app that launched Python.

        Args:
            json (str): The XMS time settings JSON string.
        """
        self.zero_time = None  # Currently not defined in GMS and always defined in SMS
        self.ref_time = None  # Dataset reference time, if there is one.
        self.use_abs_times = True
        self._abs_date_format = FOR_ABS_DATE_REGIONAL
        self._abs_time_format = FOR_ABS_TIME_REGIONAL
        self.abs_specifier = ''  # For use with strftime(), both date and time portions
        self.date_specifier = ''  # For use with strftime(), date portion only
        self.time_specifier = ''  # For use with strftime(), time portion only
        self.qt_abs_specifier = ''  # For use with QDateTime/QDateTimeEdit, both date and time portions
        self.qt_date_specifier = ''  # For use with QDateTime/QDateTimeEdit, date portion only
        self.qt_time_specifier = ''  # For use with QDateTime/QDateTimeEdit, time portion only
        self.rel_format = FOR_REL_days_s_hh_c_mm_c_ss
        self.rel_prec = None
        self._initialize(json)

    def _initialize(self, json):
        """Parse XMS time setting from JSON string."""
        xms_settings = orjson.loads(json)
        self._abs_date_format = int(xms_settings.get(KEY_ABS_DATE_FORMAT, self._abs_date_format))
        self._abs_time_format = int(xms_settings.get(KEY_ABS_TIME_FORMAT, self._abs_time_format))
        self.rel_format = int(xms_settings.get(KEY_REL_FORMAT, self.rel_format))
        display_ref_type = int(xms_settings.get(KEY_DISPLAY_REF_TYPE, DISPLAY_RELATIVE_TIMES))
        if display_ref_type == DISPLAY_RELATIVE_TIMES:
            self.use_abs_times = False
        # GMS does not have a zero time. To convert relative times to absolute timestamps when there is no zero time,
        # self.ref_time must be set after construction.
        if 'ZeroTime' in xms_settings and xms_settings['ZeroTime']:
            # Convert day offset from MFC COleDateTime epoch to Python datetime
            self.zero_time = coledatetime_to_datetime(float(xms_settings['ZeroTime']))
        # Switch from C locale to system default, persists for process? I think this is frowned upon.
        locale.setlocale(locale.LC_ALL, '')
        self._build_abs_specifier(True)
        self._build_abs_specifier(False)

    def _build_abs_specifier(self, date):
        """Initialize the absolute datetime format specifier based on current user preference in SMS.

        Args:
            date (bool): True if the date portion of the datetime, False for the time portion
        """
        if date:  # Build date portion
            if self._abs_date_format == FOR_ABS_DATE_REGIONAL:
                date_format = '%x'  # Use system locale formatting
            else:  # Use absolute date format specified in XMS
                date_format = ABS_DATE_FORMAT_SPECIFIERS[self._abs_date_format]
            self.date_specifier = date_format
            self.qt_date_specifier = convert_strftime_specifiers_to_qt(date_format, True)
        else:  # Build time portion
            if self._abs_time_format == FOR_ABS_TIME_REGIONAL:
                time_format = '%X'  # Use system locale formatting
            else:  # Use absolute time format specified in XMS
                time_format = ABS_TIME_FORMAT_SPECIFIERS[self._abs_time_format]
            self.time_specifier = time_format
            self.qt_time_specifier = convert_strftime_specifiers_to_qt(time_format, False)
        self.abs_specifier = f'{self.date_specifier} {self.time_specifier}'
        self.qt_abs_specifier = f'{self.qt_date_specifier} {self.qt_time_specifier}'

    def _format_timestamp(self, dt):
        """Format an absolute timestamp.

        If current XMS time setting is to display relative times, offset will be computed as the difference of the
        current zero time and dt (positive or negative).

        Args:
            dt (datetime): The absolute time to format

        Returns:
            (str): The formatted relative (or absolute) datetime string
        """
        if not self.use_abs_times:  # Convert to relative from zero time
            if self.zero_time is not None:  # If we have a zero time, use the difference as relative offset.
                return self._format_timespan(dt - self.zero_time)
            elif self.ref_time is not None:  # If no zero time, use difference of reference time as relative offset.
                return self._format_timespan(dt - self.ref_time)
        return dt.strftime(self.abs_specifier)

    def _format_timespan(self, time_span):  # noqa: C901
        """Format a time offset from the current zero time.

        If current XMS time setting is to display absolute times, timestamp will be constructed as the current
        zero time plus time_span (positive or negative).

        Args:
            time_span (timedelta): The relative time span to format

        Returns:
            (str): The formatted relative (or absolute) datetime string
        """
        if self.use_abs_times:  # Convert to absolute datetime that is offset from the zero time by time_span
            if self.zero_time is not None:  # Add time span to zero time if it is defined to build absolute timestamp
                abs_time = self.zero_time + time_span
            elif self.ref_time is not None:  # Add time span to ref time if it is defined to build absolute timestamp
                abs_time = self.ref_time + time_span
            else:  # Not able to build an absolute timestamp for this time span. Format as relative.
                abs_time = None

            if abs_time is not None:
                return self._format_timestamp(abs_time)

        sign = '-' if time_span.days < 0 else ''
        secs = abs(time_span).total_seconds()
        if self.rel_format < FOR_REL_days_decimal:
            days, rem = divmod(secs, 86400)  # Seconds per day: 24 * 60 * 60
            hours, rem = divmod(rem, 3600)  # Seconds per hour: 60 * 60
            mins, secs = divmod(rem, 60)
            if self.rel_format == FOR_REL_days_s_hh_c_mm_c_ss:
                return f'{sign}{int(days)} {int(hours):0>2d}:{int(mins):0>2d}:{int(secs):0>2d}'
            elif self.rel_format == FOR_REL_hours_c_mm_c_ss:
                total_hours = (days * 24) + hours
                return f'{sign}{int(total_hours)}:{int(mins):0>2d}:{int(secs):0>2d}'
            elif self.rel_format == FOR_REL_days_s_hh_c_mm:
                return f'{sign}{int(days)} {int(hours):0>2d}:{int(mins):0>2d}'
            else:  # FOR_REL_hours_c_mm:
                total_hours = (days * 24) + hours
                return f'{sign}{int(total_hours)}:{int(mins):0>2d}'
        else:  # Single unit formats
            rel_time = secs
            if self.rel_format == FOR_REL_days_decimal:
                rel_time = rel_time / 86400.0
            elif self.rel_format == FOR_REL_hours_decimal:
                rel_time = rel_time / 3600.0
            elif self.rel_format == FOR_REL_min_decimal:
                rel_time = rel_time / 60.0
            elif self.rel_format == FOR_REL_years_decimal:
                rel_time = rel_time / (86400.0 * 365.2422)
            if self.rel_prec is None:  # Use default python string formatting
                return f'{sign}{rel_time}'
            else:  # Specify the precision
                return f'{sign}{rel_time:.{self.rel_prec}f}'

    @property
    def abs_date_format(self):
        """Getter for the absolute date format enum."""
        return self._abs_date_format

    @abs_date_format.setter
    def abs_date_format(self, format_enum):
        """Setter for the absolute date format enum."""
        self._abs_date_format = format_enum
        self._build_abs_specifier(True)

    @property
    def abs_time_format(self):
        """Getter for the absolute time format enum."""
        return self._abs_time_format

    @abs_time_format.setter
    def abs_time_format(self, format_enum):
        """Setter for the absolute time format enum."""
        self._abs_time_format = format_enum
        self._build_abs_specifier(False)

    def format_time(self, time_or_span):
        """Format a time using the XMS global time settings.

        Args:
            time_or_span (timedelta or datetime): The time to format. If a timedelta,
                will be treated as an offset from the current zero time. Useful for dataset time steps that do not
                have a reference time. If dataset time step has a reference time, pass a datetime
        """
        if isinstance(time_or_span, timedelta):
            return self._format_timespan(time_or_span)
        elif isinstance(time_or_span, datetime):
            return self._format_timestamp(time_or_span)
        else:
            raise TypeError(
                'Must pass an absolute timestamp of type datetime or a relative time span of type: '
                'timedelta.'
            )
