"""Flyout widget for selecting colors."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QSignalMapper, QSize, Qt, Signal, Slot
from PySide2.QtGui import QColor, QIcon, QPixmap
from PySide2.QtWidgets import QFrame, QGridLayout, QPushButton, QVBoxLayout

# 3. Aquaveo modules

# 4. Local modules


class ColorPopout(QFrame):
    """A widget for showing colors that the user can select. Also has a button to get to the color dialog."""
    color_changed = Signal(QColor)
    more_colors_clicked = Signal()

    def __init__(self, pos, parent=None):
        """Initializes the widget.

        Args:
            pos (QPoint): Location to display color popout.
            parent (Something derived from QFrame): The parent window.
        """
        super().__init__(parent)
        self.setWindowFlags(self.windowFlags() | Qt.Popup)
        self.colors = []
        self.sig_map = QSignalMapper()
        all_layout = QVBoxLayout(self)
        layout = QGridLayout()

        layout.setMargin(1)
        layout.setSpacing(1)

        self.row = 0
        self.col = 0

        self.add_color(QColor(0, 0, 0), layout)
        self.add_color(QColor(170, 0, 0), layout)
        self.add_color(QColor(0, 85, 0), layout)
        self.add_color(QColor(170, 85, 0), layout)
        self.add_color(QColor(0, 170, 0), layout)
        self.add_color(QColor(170, 170, 0), layout)
        self.add_color(QColor(0, 255, 0), layout)
        self.add_color(QColor(170, 255, 0), layout)
        self.add_color(QColor(0, 0, 127), layout)
        self.add_color(QColor(170, 0, 127), layout)
        self.add_color(QColor(0, 85, 127), layout)
        self.add_color(QColor(170, 85, 127), layout)
        self.add_color(QColor(0, 170, 127), layout)
        self.add_color(QColor(170, 170, 127), layout)
        self.add_color(QColor(0, 255, 127), layout)
        self.add_color(QColor(170, 255, 127), layout)
        self.add_color(QColor(0, 0, 255), layout)
        self.add_color(QColor(170, 0, 255), layout)
        self.add_color(QColor(0, 85, 255), layout)
        self.add_color(QColor(170, 85, 255), layout)
        self.add_color(QColor(0, 170, 255), layout)
        self.add_color(QColor(170, 170, 255), layout)
        self.add_color(QColor(0, 255, 255), layout)
        self.add_color(QColor(170, 255, 255), layout)
        self.add_color(QColor(85, 0, 0), layout)
        self.add_color(QColor(255, 0, 0), layout)
        self.add_color(QColor(85, 85, 0), layout)
        self.add_color(QColor(255, 85, 0), layout)
        self.add_color(QColor(85, 170, 0), layout)
        self.add_color(QColor(255, 170, 0), layout)
        self.add_color(QColor(85, 255, 0), layout)
        self.add_color(QColor(255, 255, 0), layout)
        self.add_color(QColor(85, 0, 127), layout)
        self.add_color(QColor(255, 0, 127), layout)
        self.add_color(QColor(85, 85, 127), layout)
        self.add_color(QColor(255, 85, 127), layout)
        self.add_color(QColor(85, 170, 127), layout)
        self.add_color(QColor(255, 170, 127), layout)
        self.add_color(QColor(85, 255, 127), layout)
        self.add_color(QColor(255, 255, 127), layout)
        self.add_color(QColor(85, 0, 255), layout)
        self.add_color(QColor(255, 0, 255), layout)
        self.add_color(QColor(85, 85, 255), layout)
        self.add_color(QColor(255, 85, 255), layout)
        self.add_color(QColor(85, 170, 255), layout)
        self.add_color(QColor(255, 170, 255), layout)
        self.add_color(QColor(85, 255, 255), layout)
        self.add_color(QColor(255, 255, 255), layout)

        self.sig_map.mapped[int].connect(self.color_clicked)

        all_layout.addLayout(layout)
        dlg_btn = QPushButton(self)
        dlg_btn.setText("More Colors...")
        dlg_btn.clicked.connect(self.more_colors_clicked)
        all_layout.addWidget(dlg_btn)

        self.setLayout(all_layout)
        self.setFrameShape(QFrame.Box)

        new_y_pos = pos.y()
        self.move(pos.x(), new_y_pos)

    @Slot()
    def color_clicked(self, color_idx):
        """Called when a color is clicked. Emits color_changed.

        Args:
            color_idx (int): Index of the color clicked
        """
        self.color_changed.emit(self.colors[color_idx])

    def add_color(self, new_color, layout):
        """Adds a color to the popout.

        Args:
            new_color (QColor): The new color to add.
            layout (QGridLayout): The layout to add the button with the new color to.
        """
        # 9 is awfully small. 12 better matches the old MFC
        icon_length = 12
        k_margin = 3

        btn = QPushButton()
        btn.setStyleSheet("* { border-style: inset; }")

        icon = QIcon()
        size = QSize(icon_length, icon_length)
        pix = QPixmap(size)
        pix_disabled = QPixmap(size)
        color = new_color
        disabled_color = QColor(0, 0, 0, 0)
        # always draw the icon opaque
        color.setAlpha(255)
        pix.fill(color)
        pix_disabled.fill(disabled_color)

        icon.addPixmap(pix, QIcon.Active)
        icon.addPixmap(pix, QIcon.Normal)
        icon.addPixmap(pix, QIcon.Selected)
        icon.addPixmap(pix_disabled, QIcon.Disabled)
        btn.setFixedSize(icon_length + k_margin, icon_length + k_margin)
        btn.setIcon(icon)

        btn.clicked.connect(self.sig_map.map)
        color_idx = self.row * 8 + self.col
        self.sig_map.setMapping(btn, color_idx)
        self.colors.append(new_color)

        layout.addWidget(btn, self.row, self.col)
        self.col += 1
        # 8 = number of colors per row
        if self.col == 8:
            self.col = 0
            self.row += 1
