"""Factory for getting button icons for display options."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QIcon, QImage, QPainter, QPen, QPixmap

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.data.line_style import LineOptions, LineStyle
from xms.guipy.data.point_symbol import PointOptions
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.widgets.point_symbol_widget import PointSymbolWidget
from xms.guipy.widgets.polygon_texture_widget import PolygonTextureWidget


class DisplayOptionIconFactory:
    """Factory for getting button icons for display options."""
    @staticmethod
    def get_icon(option, btn_size):
        """Get an icon for a display options selector button.

        Args:
            option (LineOptions or PointOptions or PolygonOptions): The display options to get icon for
            btn_size (QSize): Size of the QPushButton

        Returns:
            (QIcon): The icon for the button

        """
        if isinstance(option, LineOptions):
            draw_img = QImage(btn_size, btn_size, QImage.Format_ARGB32_Premultiplied)
            draw_img.fill(0x00000000)
            draw_painter = QPainter(draw_img)
            draw_pen = QPen()
            draw_pen.setColor(option.color)
            if option.style == LineStyle.SOLIDLINE:
                draw_pen.setStyle(Qt.SolidLine)
                draw_pen.setWidth(option.width)
            elif option.style == LineStyle.DASHEDLINE:
                draw_pen.setStyle(Qt.DashLine)
                draw_pen.setWidth(1)
            draw_painter.setPen(draw_pen)
            mid_btn = btn_size / 2
            draw_painter.drawLine(0, mid_btn, btn_size, mid_btn)
            draw_painter.end()
            draw_pix = QPixmap.fromImage(draw_img)
            draw_icon = QIcon(draw_pix)
            return draw_icon
        elif isinstance(option, PointOptions):
            return PointSymbolWidget.get_icon(
                PointSymbolWidget.symbol_svg[option.symbol], option.color, option.size, btn_size
            )
        elif isinstance(option, PolygonOptions):
            svg = PolygonTextureWidget.texture_svg[option.texture]
            return PolygonTextureWidget.get_icon(svg, option.color, btn_size)
        else:
            raise Exception(f'Unknown display option type {type(option)}.')
