"""Widget for the polygon texture selector."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QSize, Qt
from PySide2.QtGui import QColor, QIcon, QImage, QPainter, QPixmap
from PySide2.QtSvg import QGraphicsSvgItem
from PySide2.QtWidgets import QGridLayout, QPushButton, QSizePolicy

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.data.polygon_texture import PolygonTexture
import xms.guipy.resources.guipy  # noqa: F401, F403
from xms.guipy.widgets.button_enum import ButtonEnumWidget


class PolygonTextureWidget(ButtonEnumWidget):
    """A widget showing polygon textures on buttons in a grid in a given color."""
    texture_svg = {
        PolygonTexture.solid_pattern: ":resources/textures/texture1.svg",
        PolygonTexture.diagonal_cross_pattern: ":resources/textures/texture2.svg",
        PolygonTexture.cross_pattern: ":resources/textures/texture3.svg",
        PolygonTexture.diagonal_right_pattern: ":resources/textures/texture4.svg",
        PolygonTexture.diagonal_left_pattern: ":resources/textures/texture5.svg",
        PolygonTexture.vertical_pattern: ":resources/textures/texture6.svg",
        PolygonTexture.igneous2_pattern: ":resources/textures/texture7.svg",
        PolygonTexture.conglomerate_pattern: ":resources/textures/texture8.svg",
        PolygonTexture.shale_pattern: ":resources/textures/texture9.svg",
        PolygonTexture.limestone_pattern: ":resources/textures/texture10.svg",
        PolygonTexture.granite_pattern: ":resources/textures/texture11.svg",
        PolygonTexture.quartzite_pattern: ":resources/textures/texture12.svg",
        PolygonTexture.sandstone_pattern: ":resources/textures/texture13.svg",
        PolygonTexture.tuff_pattern: ":resources/textures/texture14.svg",
        PolygonTexture.polymorphic_pattern: ":resources/textures/texture15.svg",
        PolygonTexture.sand_shale_pattern: ":resources/textures/texture16.svg",
        PolygonTexture.dolomite_pattern: ":resources/textures/texture17.svg",
        PolygonTexture.lava_pattern: ":resources/textures/texture18.svg",
        PolygonTexture.igneous_pattern: ":resources/textures/texture19.svg",
        PolygonTexture.clay_pattern: ":resources/textures/texture20.svg",
        PolygonTexture.clay_invert_pattern: ":resources/textures/texture21.svg",
        PolygonTexture.stripes_pattern: ":resources/textures/texture22.svg",
        PolygonTexture.rays_pattern: ":resources/textures/texture23.svg",
        PolygonTexture.null_pattern: ":resources/textures/texture1.svg",
    }

    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (Something derived from QObject): The parent object.
        """
        super().__init__(parent)
        self.layout = QGridLayout(self)
        self.buttons = {}
        row = 0
        col = 0
        for texture in self.texture_svg:
            btn = QPushButton(self)
            btn.setMinimumSize(QSize(50, 50))
            btn.setMaximumSize(QSize(150, 150))
            btn.setSizeIncrement(1, 1)
            btn.setCheckable(True)
            btn.setAutoExclusive(True)
            self.layout.addWidget(btn, row, col)
            self.buttons[texture] = btn
            col += 1
            if col > 4:
                col = 0
                row += 1

        self.setSizePolicy(QSizePolicy.Policy.Preferred, QSizePolicy.Policy.Preferred)

    def set_texture(self, texture):
        """Checks the button showing the texture.

        Args:
            texture (PolygonTexture): The texture that is to be checked.
        """
        self.buttons[texture].setChecked(True)

    def get_texture(self):
        """Gets the texture of the currently checked button.

        Returns:
            texture (PolygonTexture): The texture.
        """
        for texture in self.texture_svg:
            if self.buttons[texture].isChecked():
                return texture

    def get_current_icon(self, color, btn_size):
        """Gets the icon drawn from the current texture, in the specified color.

        Args:
            color (QColor): The current color.
            btn_size (int): The size of the button to draw on. Assumes square.

        Returns:
            icon (QIcon): The icon.
        """
        return self.get_icon(self.texture_svg[self.get_symbol()], color, btn_size)

    @staticmethod
    def get_icon(path, color, btn_size):
        """Gets the icon drawn from the svg file, in the specified color.

        Args:
            path (str): Relative path to the *.svg resource file.
            color (QColor): The current color.
            btn_size (int): The size of the button to draw on. Assumes square.

        Returns:
            icon (QIcon): The icon.
        """
        svg = QGraphicsSvgItem(path)
        svg_img = QImage(1000, 1000, QImage.Format_ARGB32_Premultiplied)
        svg_img.fill(0x00000000)
        svg_painter = QPainter(svg_img)
        svg.renderer().render(svg_painter)
        svg_painter.end()
        svg_pix = QPixmap.fromImage(svg_img.scaled(btn_size, btn_size))
        if path.endswith('21.svg'):
            mask = svg_pix.createMaskFromColor(QColor(1, 1, 1, 255), Qt.MaskInColor)
        else:
            mask = svg_pix.createMaskFromColor(QColor(1, 1, 1, 255), Qt.MaskOutColor)
        svg_pix.fill(color, btn_size)
        svg_pix.setMask(mask)
        svg_icon = QIcon(svg_pix)
        return svg_icon

    def draw_all_textures(self):
        """Draws all of the different textures on buttons, in the black."""
        color = QColor('black')
        icon_size = QSize(40, 40)
        for texture in self.texture_svg:
            size = self.buttons[PolygonTexture.null_pattern].size().width()
            self.buttons[texture].setIcon(self.get_icon(self.texture_svg[texture], color, size))
            self.buttons[texture].setIconSize(icon_size)
