"""Model checker for DMI simulations."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ModelCheckError, Query
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file

# 4. Local modules
from xms.hecras._TopologyCreatorFromDatasets import TopologyCreatorFromDatasets


class HECRASModelChecker:
    """Model checker for DMI simulations."""

    def __init__(self, xms_data=None):
        """Constructor.

        xms_data (dict): Data supplied for testing
        """
        self.errors = []
        self._query = None
        self._do_ugrid = None
        if xms_data:  # Testing
            self._do_ugrid = xms_data['do_ugrid']
        else:  # pragma: no cover
            self._get_xms_data()  # Retrieve simulation mesh from XMS

    def _get_xms_data(self):  # pragma: no cover
        """Retrieve simulation data from XMS."""
        self._query = Query()
        sim_uuid = self._query.current_item_uuid()
        sim_item = tree_util.find_tree_node_by_uuid(self._query.project_tree, sim_uuid)
        ugrid_items = tree_util.descendants_of_type(sim_item, xms_types=['TI_UGRID_PTR'], allow_pointers=True,
                                                    recurse=False)
        if ugrid_items:
            self._do_ugrid = self._query.item_with_uuid(ugrid_items[0].uuid)

    def check_mesh(self):
        """Checks for voids in the mesh."""
        if not self._do_ugrid:
            model_error = ModelCheckError()
            model_error.problem_text = 'STOP! No HEC-RAS domain mesh defined.'
            model_error.description_text = 'A mesh geometry must be linked to the simulation before exporting.'
            model_error.fix_text = 'Create a 2D Mesh and drag it under the simulation item.'
            self.errors.append(model_error)
            return

        # Read the mesh from the CoGrid file
        co_grid = read_grid_from_file(self._do_ugrid.cogrid_file)
        ugrid = co_grid.ugrid

        # get polys
        polys = [
            [cell_idx + 1, [pt_idx + 1 for pt_idx in ugrid.get_cell_points(cell_idx)]]
            for cell_idx in range(ugrid.cell_count)
        ]

        # get points
        locs = ugrid.locations
        points = [[i + 1, [locs[i][0], locs[i][1]]] for i in range(len(locs))]

        try:
            horiz_units = "FOOT" if self._do_ugrid.projection.horizontal_units.startswith('FEET') else 'SI Units'
            topo = TopologyCreatorFromDatasets(polys, points, horiz_units)
            topo.process()
        except RuntimeError:
            model_error = ModelCheckError()
            model_error.problem_text = 'Voids detected in the domain.'
            model_error.description_text = 'HEC-RAS does not support voids in the domain.'
            model_error.fix_text = 'Remove all voids in the domain.'
            self.errors.append(model_error)

    def run_checks(self):
        """Runs the check_mesh function and sends the errors to SMS."""
        self.check_mesh()

        if self.errors and self._query:  # pragma: no cover
            self._query.add_model_check_errors(self.errors)
            self._query.send()
