"""This object reads formats .2dm, .2dm-extended, and .rasgeo.

create_empty_lists()
    It creates empty lists for nodelinks and node_polygons_links to be populated later

reorient_polygons()
    It re-orients the vertices of polygons, so all of them are counter-clockwise as required by RAS2D

populate_links()
    It is the most complex of all the methods of this object
    It populates the links (lines/faces)
    It collapses the duplicated links
        Since the polygons are oriented, we can assume that each link will appear in opposite directions
        I create two identical list of links and then sort them differently.
            The first list is sorted by node_i and then by node_f
            The second list is sorted by node_f and then by node_i
        I compare element by element of both sorted lists: if node_i == node_f and node_f == node_i
        If they are equal it means that they are a duplicated link.
        If the element in the first list has a lower original index than the second, I add it to the list of unique
            links. "It came first"
        If the element in the second list has a lower original index than the first, I ignore it. I will face that link
            again later.
        If the elements are not equal, I increase the corresponding index. I move along one of the lists until it
            matches.
        In all this process I have to keep track of the original numbering of the links, so I'm populating a look up
            table [old_index, new_index] that I will need later.
    I go through the nodes replacing the original link numbering by the new numbering, and updating the direction of the
        modified links.
    I go through the polygons replacing the original link numbering by the new numbering, and updating the direction of
        the modified links.
    It identifies the single links as boundary links

create_boundary_polygons()
    Create boundary polygons

find_cell_and_link_sharing_node()
    Populate cells and links surrounding each node

find_neighboring_cells()
    Sort neighboring cells for each node

calculate_centers()
    Calculate centers

orient_neighboring_cells()
    Orient neighboring cells for each node

extract_boundary()
    Find boundary links (one adjacent polygon)
    Find boundary as chain of boundary links

populate_topology()
    Populate the topology attributes of the object.
    All the indexes are decreased by one, because the rasgeo is base 1  and the hdf is base 0
"""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.hecras._OrientationChecker import OrientationChecker


def calculate_area(points):
    """Calculates area enclosed by points.

    Args:
        points (:obj:`list` of :obj:`xms.data_objects.Parameters.Point`): Points of the polygon, in order

    Returns:
        float: Area, always positive.
    """
    points.append(points[0])  # To close the polygon
    sumarea = 0
    for i in range(0, len(points) - 1):
        p0 = points[i]
        p1 = points[i + 1]
        x0 = p0[0]
        y0 = p0[1]
        x1 = p1[0]
        y1 = p1[1]
        area = 0.5 * (y0 + y1) * float(x1 - x0)
        sumarea = sumarea + area
    return abs(sumarea)


class TopologyCreatorFromDatasets:
    """This object reads formats .2dm, .2dm-extended, and .rasgeo."""
    nodes = None
    links = None
    rawpolygons = None
    polygons = None
    polygons_area = None
    orientedpolygons = None
    node_polygons_links = None  # Ex [957, [1496, 1497, 1538, 1586, 1588, 1590, 1589]]
    polygon_links = None
    nodelinks = None
    realcenters = None
    centers = None
    boundary = None
    sorted_polygons_list = None
    units = "SI Units"  # By default
    # Processed topologies (with all the indexes decreased)
    topo_vertices = None
    topo_link_nodes = None
    topo_link_polygons = None
    topo_centers = None
    topo_boundary = None
    topo_bound_in_link = None
    topo_perimeter = None
    topo_vertices_polygons = None
    topo_polygon_links = None
    topo_vertices_links = None

    def __init__(self, rawpolygons, nodes, units):
        """Constructor.

        Args:
            rawpolygons (list): Polygons of the geometry
            nodes (list): Locations of the geometry points
            units (str): The geometry's horizontal projection units
        """
        self.rawpolygons = rawpolygons
        self.nodes = nodes
        self.units = units

    def process(self):
        """Formatted Printer and Parent function for the Class."""
        print("        =================================")
        print("        Extracting topology from 2dm file")
        print("        =================================")
        self.create_empty_lists()
        self.reorient_polygons()
        self.populate_links()
        self.create_boundary_polygons()
        self.find_cell_and_link_sharing_node()
        self.find_neighboring_cells()
        self.calculate_centers()
        self.orient_neighboring_cells()
        self.extract_boundary()
        self.populate_topology()
        print("        ==================")
        print("        Topology extracted")
        print("        ==================")

    def create_empty_lists(self):
        """Creates empty lists for nodelinks and node_polygons_links to be populated later."""
        nodeqty = len(self.nodes)
        nodelinks = []
        node_polygons_links = []
        for i in range(0, nodeqty):
            nodelinks.append([self.nodes[i][0], []])
            node_polygons_links.append([self.nodes[i][0], [], []])
        self.rawpolygons.sort()
        self.nodes.sort()
        self.nodelinks = nodelinks
        self.node_polygons_links = node_polygons_links

    def reorient_polygons(self):
        """Re-orients the vertices of polygons, so all of them are counter-clockwise as required by RAS2D."""
        och = OrientationChecker()
        polygons = []
        polygons_area = []
        for poly in self.rawpolygons:
            number = poly[0]
            vertices = poly[1]
            points = []
            for vertex in vertices:
                node = self.nodes[vertex - 1]  # The numbers start at 1
                if node[0] != vertex:
                    print("Problem at polygon #" + str(number))
                coords = node[1]
                points.append(coords)
            isclockwise = och.is_clockwise(points)
            area = calculate_area(points)
            if isclockwise is True:
                vertices.reverse()
            elif isclockwise is False:
                pass
            elif isclockwise is None:
                print("8-shaped polygon #" + str(number))
            polygons.append([number, vertices])
            polygons_area.append(area)
        self.polygons = polygons
        self.polygons_area = polygons_area

    def populate_links(self):
        """Populates the links for lines and faces.

        It is the most complex of all the methods of this object
        It populates the links (lines/faces)
        It collapses the duplicated links
            Since the polygons are oriented, we can assume that each link will appear in opposite directions
            I create two identical list of links and then sort them differently.
                The first list is sorted by node_i and then by node_f
                The second list is sorted by node_f and then by node_i
            I compare element by element of both sorted lists: if node_i == node_f and node_f == node_i
            If they are equal it means that they are a duplicated link.
            If the element in the first list has a lower original index than the second, I add it to the list of unique
                links. "It came first"
            If the element in the second list has a lower original index than the first, I ignore it. I will face that
                link again later.
            If the elements are not equal, I increase the corresponding index. I move along one of the lists until it
                matches.
            In all this process I have to keep track of the original numbering of the links, so I'm populating a look
                up table [old_index, new_index] that I will need later.
        I go through the nodes replacing the original link numbering by the new numbering, and updating the direction
            of the modified links.
        I go through the polygons replacing the original link numbering by the new numbering, and updating the direction
            of the modified links.
        It identifies the single links as boundary links.
        """
        polygons = self.polygons
        nodelinks = self.nodelinks
        node_templinks = nodelinks  # nodelinks = [[node_i, [[link_1, direction_1], ..., [link_n, direction_n]], ...]
        templinks_a = []
        templinks_b = []
        templink_i = 1  # Index for the templinks
        templinks_lookup = []
        polygons_templinks = []
        for polygon in polygons:  # polygon = [poly_i, [node_1, ..., node_n]]
            poly_i = polygon[0]
            nodes = polygon[1]
            nodes = nodes + [nodes[0]]  # Add the first element at last place to close the polygon
            polygon_templinks = []
            # polygon_templink = [poly_i, [[templink_1, tempdirection_1], ...,
            # [templink_n, tempdirection_n]] tempdirections are 1

            for i in range(0, len(nodes) - 1):  # I populate templinks with all the links, even if they are repeated
                node_i = nodes[i]
                node_f = nodes[i + 1]
                templink = [templink_i, [node_i, node_f], [poly_i, None], None]
                # templink = [templink_i, [node_i, node_f], [poly_right, poly_left],
                #             inboundary] << right and left or left and right??

                # templink_lookup = [templink_i, new_link_index, direction]
                templink_lookup = [templink_i, None, None]
                polygon_templinks.append([templink_i, 1])  # I'll reverse it later if necessary
                node_templinks[node_i - 1][1].append([templink_i, 1])  # Need to add node_f as well???
                node_templinks[node_f - 1][1].append([templink_i, -1])  # Need to add node_f as well???
                templink_i = templink_i + 1
                templinks_a.append(templink)
                templinks_b.append(templink)
                templinks_lookup.append(templink_lookup)
            polygons_templinks.append([poly_i, polygon_templinks])
        # Now I need the two copies sorted differently to perform a paired comparison
        templinks_a.sort(key=lambda elem: (elem[1][0], elem[1][1]))
        templinks_b.sort(key=lambda elem: (elem[1][1], elem[1][0]))
        links = []
        link_i = 1
        i = 0
        j = 0
        keepin = True
        while keepin:
            # print(i, j)
            templink_a = templinks_a[i]
            templink_i_a = templink_a[0]
            nodes_a = templink_a[1]
            polygons_a = templink_a[2]
            templink_b = templinks_b[j]
            templink_i_b = templink_b[0]
            nodes_b = templink_b[1]
            polygons_b = templink_b[2]
            if nodes_a[0] == nodes_b[1] and nodes_a[1] == nodes_b[0]:
                # They are the same link but in opposite directions
                if polygons_b[0] > polygons_a[0]:       # Who was defined first?
                    poly_right = polygons_a[0]
                    poly_left = polygons_b[0]
                    node_i = nodes_a[0]
                    node_f = nodes_a[1]
                    inboundary = 0
                    link = [link_i, [node_i, node_f], [poly_right, poly_left], inboundary]
                    links.append(link)      # Populate links
                    templinks_lookup[templink_i_a - 1][1] = link_i   # Update lookup
                    templinks_lookup[templink_i_a - 1][2] = 1        # Update lookup
                    templinks_lookup[templink_i_b - 1][1] = link_i   # Update lookup
                    templinks_lookup[templink_i_b - 1][2] = -1       # Update lookup
                    # Increase index
                    link_i = link_i + 1
                # Increase both search indexes
                i = i + 1
                j = j + 1
            elif nodes_b[1] > nodes_a[0] or (nodes_b[1] == nodes_a[0] and nodes_b[0] > nodes_a[1]):
                # If in templinks_a but not in templinks_b, add and advance i
                poly_right = polygons_a[0]
                poly_left = None
                node_i = nodes_a[0]
                node_f = nodes_a[1]
                inboundary = 1
                link = [link_i, [node_i, node_f], [poly_right, poly_left], inboundary]
                links.append(link)      # Populate links
                templinks_lookup[templink_i_a - 1][1] = link_i      # Update lookup
                templinks_lookup[templink_i_a - 1][2] = 1       # Update lookup
                # Increase index
                link_i = link_i + 1
                # Increase i
                i = i + 1
            elif nodes_b[1] < nodes_a[0] or (nodes_b[1] == nodes_a[0] and nodes_b[0] < nodes_a[1]):
                # If in templinks_b but not in templinks_a, just advance j
                j = j + 1
            else:
                print("Should never get here")
            if i > len(templinks_a) - 1 or j > len(templinks_b) - 1:        # Leave the while loop
                keepin = False
        if i < len(templinks_a):
            # Add remaining as boundaries (sure?) Could finish a and not b, and if so,
            # would it worth to keep on looking at it?
            for k in range(i, len(templinks_a)):
                templink_a = templinks_a[k]
                templink_i_a = templink_a[0]
                nodes_a = templink_a[1]
                polygons_a = templink_a[2]
                poly_right = polygons_a[0]
                poly_left = None
                node_i = nodes_a[0]
                node_f = nodes_a[1]
                inboundary = 1
                link = [link_i, [node_i, node_f], [poly_right, poly_left], inboundary]
                links.append(link)      # Populate links
                templinks_lookup[templink_i_a - 1][1] = link_i       # Update lookup
                templinks_lookup[templink_i_a - 1][2] = 1        # Update lookup
                # Increase index
                link_i = link_i + 1
                # Increase i
                i = i + 1
        # Now I need to renumber polygon_templinks into polygon_links.
        # It is important to keep the original order, because it was counterclockwise.
        polygon_links = []
        for polygon in polygons_templinks:
            poly_i = polygon[0]
            templinks = polygon[1]
            plinks = []
            for templink in templinks:
                tempdirection = templink[1]     # Should be always 1
                plink = templinks_lookup[templink[0] - 1][1]
                newdirection = templinks_lookup[templink[0] - 1][2]
                direction = newdirection * tempdirection
                plinks.append([plink, direction])
            polygon_links.append([poly_i, plinks])
        # Now in need to renumber node_templinks into nodelinks.
        nodelinks = []      # nodelinks = [[node_i, [[link_1, direction_1], ..., [link_n, direction_n]], ...]
        for node in node_templinks:
            node_i = node[0]
            templinks = node[1]
            nlinks = []
            for templink in templinks:
                tempdirection = templink[1]
                nlink = templinks_lookup[templink[0] - 1][1]
                newdirection = templinks_lookup[templink[0] - 1][2]
                direction = newdirection * tempdirection
                if not [nlink, direction] in nlinks:
                    nlinks.append([nlink, direction])
            nodelinks.append([node_i, nlinks])
        self.links = links
        self.polygon_links = polygon_links
        self.nodelinks = nodelinks

    def create_boundary_polygons(self):
        """Create boundary polygons."""
        links = []
        polygons = self.polygons
        number = len(polygons) + 1
        for link in self.links:
            if link[3] == 1:
                link[2][1] = number
                polygons.append([number, link[1]])
                self.polygon_links.append([number, [[link[0], -1]]])        # <<<<<<<<<<<<<<Not sure
                number = number + 1
            links.append(link)
        self.links = links
        self.polygons = polygons

    def find_cell_and_link_sharing_node(self):
        """Populate cells and links surrounding each node."""
        for link in self.links:
            # [link_index, [node_i, node_f], [left_poly, right_poly], boundary]
            link_index = link[0]
            nodes = link[1]
            polygons = link[2]
            for node in nodes:
                linksthere = self.node_polygons_links[node - 1][2]
                if link_index not in linksthere:
                    linksthere.append(link_index)
                    self.node_polygons_links[node - 1][2] = linksthere
                for poly in polygons:
                    polysthere = self.node_polygons_links[node - 1][1]
                    if poly not in polysthere:
                        polysthere.append(poly)
                        self.node_polygons_links[node - 1][1] = polysthere

    def find_neighboring_cells(self):
        """Sort neighboring cells for each node."""
        sorted_polygons_list = []
        for node in self.node_polygons_links:
            node_index = node[0]
            polygons = node[1]
            links = node[2]
            links_polygons = []
            # Create a subset of links and their right and left polygons, extracting them from self.links.
            startinglink = 0
            for i in range(0, len(links)):
                link = links[i]
                link_polygons = self.links[link - 1][2]
                inboundary = self.links[link - 1][3]
                if inboundary == 1:
                    startinglink = i     # start from the link in the boundary
                links_polygons.append([link, link_polygons, inboundary])
            sortedpolygons = []
            active_link = links_polygons[startinglink]
            available_links = links_polygons[:startinglink] + links_polygons[startinglink + 1:]
            target_polygon = active_link[1][0]
            sortedpolygons.append(target_polygon)
            while len(sortedpolygons) < len(polygons) - 1:
                new_available_links = []
                found_one = False
                for search_link in available_links:
                    if search_link[1][0] == target_polygon and not found_one:
                        found_one = True
                        target_polygon = search_link[1][1]
                        sortedpolygons.append(target_polygon)
                    elif search_link[1][1] == target_polygon and not found_one:
                        found_one = True
                        target_polygon = search_link[1][0]
                        sortedpolygons.append(target_polygon)
                    else:
                        new_available_links.append(search_link)
                available_links = new_available_links
            # Add the last polygon
            for polygon in active_link[1]:
                if polygon not in sortedpolygons:
                    sortedpolygons.append(polygon)
            sorted_polygons_list.append([node_index, sortedpolygons])
        self.sorted_polygons_list = sorted_polygons_list

    def orient_neighboring_cells(self):
        """Orient neighboring cells for each node."""
        och = OrientationChecker()
        new_sorted_polygons_list = []
        for node in self.sorted_polygons_list:
            node_index = node[0]
            polygons = node[1]
            centers = []
            for polygon in polygons:
                # Retrieve center
                center = self.centers[polygon - 1][1]
                centers.append(center)
            isclockwise = och.is_clockwise(centers)
            if isclockwise is True:
                polygons.reverse()
            elif isclockwise is False:
                pass
            elif isclockwise is None:
                print("8-shaped node: " + str(node_index) + " for polygons " + str(polygons))
                # Keep it and see if it works...
                # ...or not keep it.
                polygons.reverse()
            new_sorted_polygons_list.append([node_index, polygons])
        self.sorted_polygons_list = new_sorted_polygons_list

    def populate_topology(self):        # Prepare the data for TopologyReader
        """Populate the topology attributes of the object.

        All the indexes are decreased by one, because the rasgeo is base 1  and the hdf is base 0.
        """
        topo_vertices = []
        topo_link_nodes = []
        topo_link_polygons = []
        topo_centers = []
        topo_boundary = []
        topo_bound_in_link = []
        topo_perimeter = []
        topo_vertices_polygons = []
        topo_polygon_links = []
        topo_vertices_links = []
        for node in self.nodes:
            topo_vertices.append([node[1][0], node[1][1]])
        for link in self.links:
            topo_link_nodes.append([link[1][0] - 1, link[1][1] - 1])
            topo_link_polygons.append([link[2][0] - 1, link[2][1] - 1])
            topo_perimeter.append(link[3])
        for center in self.centers:
            topo_centers.append([center[1][0], center[1][1]])
        for bound in self.boundary:
            topo_boundary.append([bound[1][0], bound[1][1]])
            topo_bound_in_link.append(bound[2] - 1)
        for node in self.sorted_polygons_list:
            polygons = []
            for poly in node[1]:        # Where the sorted polygons are
                polygons.append(poly - 1)
            topo_vertices_polygons.append(polygons)
        for poly in self.polygon_links:
            links = []
            for link in poly[1]:
                links.append([link[0] - 1, link[1]])
            topo_polygon_links.append([poly[0] - 1, links])
        for node in self.nodelinks:
            links = []
            for link in node[1]:
                links.append([link[0] - 1, link[1]])
            topo_vertices_links.append([node[0] - 1, links])
        self.topo_vertices = topo_vertices
        self.topo_link_nodes = topo_link_nodes
        self.topo_link_polygons = topo_link_polygons
        self.topo_centers = topo_centers
        self.topo_boundary = topo_boundary
        self.topo_bound_in_link = topo_bound_in_link
        self.topo_perimeter = topo_perimeter
        self.topo_vertices_polygons = topo_vertices_polygons
        self.topo_polygon_links = topo_polygon_links
        self.topo_vertices_links = topo_vertices_links

    def calculate_centers(self):
        """Calculate polygon centers."""
        realcenters = []
        centers = []
        for polygon in self.polygons:
            number = polygon[0]
            vertices = polygon[1]
            sumx = 0
            sumy = 0
            for vertex in vertices:
                node = self.nodes[vertex - 1]
                x = node[1][0]
                y = node[1][1]
                sumx = sumx + x
                sumy = sumy + y
            xcen = sumx / float(len(vertices))
            ycen = sumy / float(len(vertices))
            centers.append([number, [xcen, ycen]])
            if len(vertices) > 2:
                realcenters.append([number, [xcen, ycen]])
            else:
                pass
        self.centers = centers
        self.realcenters = realcenters

    def extract_boundary(self):
        """Find boundary links (one adjacent polygon). Find boundary as chain of boundary links."""
        # First filter the links
        boundarylinks = []
        chainedlinks = []
        for link in self.links:
            if link[3] == 1:
                boundarylinks.append(link)
        chainedlinks.append(boundarylinks[0])
        availablelinks = boundarylinks[1:]
        # Given that all the polygons are counterclockwise oriented,
        # I can assume that the match is end_node to begin_node
        searchfornode = chainedlinks[-1][1][1]
        while len(availablelinks) > 0:
            newavailablelinks = []
            keep_going = False
            for link in availablelinks:
                if link[1][0] == searchfornode:
                    chainedlinks.append(link)
                    searchfornode = chainedlinks[-1][1][1]
                    keep_going = True
                else:
                    newavailablelinks.append(link)
            if not keep_going:
                raise RuntimeError('Voids detected in the domain.')
            availablelinks = newavailablelinks
        chainedlinks.append(chainedlinks[0])        # To close the boundary
        chainedlinks.reverse()      # Not sure if necessary. All their examples is clockwise.
        boundary = []
        for i in range(0, len(chainedlinks)):
            link = chainedlinks[i]
            endnode = link[1][1]
            coords = self.nodes[endnode - 1][1]
            inlink = link[0]
            boundary.append([i + 1, coords, inlink])
        self.boundary = boundary
