"""Utility functions to deal with query."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, MenuItem, Query, XmsEnvironment
from xms.api.tree import tree_util, TreeNode
from xms.core.filesystem import filesystem
from xms.data_objects.parameters import Component
from xms.guipy import file_io_util

# 4. Local modules


def get_default_grid(query: Query) -> 'TreeNode | None':
    """Return a grid if there is just one, else None.

    Args:
        query: Object for communicating with GMS.

    Returns:
        See description.
    """
    if not query or not query.project_tree:
        return None
    grid_nodes = tree_util.descendants_of_type(query.project_tree, xms_types=['TI_UGRID'])
    return grid_nodes[0] if grid_nodes and len(grid_nodes) == 1 else None


def add_tree_menu_item(
    menu_text: str,
    method_name: str,
    main_file: str | Path,
    class_name: str,
    module_name: str,
    menu_list: list['MenuItem | None'],
    icon: str = ''
) -> None:
    """Adds a command to the menu.

    Args:
        menu_text (str): Text of the menu command.
        method_name (str): Name of method to call on the class (class_name).
        main_file (Path): File path of component main file.
        class_name (str): Name of component class.
        module_name (str): Python module name.
        menu_list (list of xmsapi.dmi.MenuItem): List of menu commands.
        icon (str): Name of the icon, like 'save.svg'.
    """
    # Create the ActionRequest
    action = ActionRequest(
        main_file=str(main_file),
        modality='modal',
        class_name=class_name,
        module_name=module_name,
        method_name=method_name,
        parameters={'main_file': str(main_file)}
    )

    # Create the MenuItem
    icon_path = icon if icon else None
    menu_item = MenuItem(text=menu_text, action=action, icon_path=icon_path)
    menu_list.append(menu_item)


def get_ugrid_filename(ugrid_uuid: str, query: Query) -> str:
    """Returns the filename where the UGrid is saved.

    Args:
        ugrid_uuid (str): uuid of the UGrid.
        query (Query): Object for communicating with GMS

    Returns:
        See description.
    """
    if not query or not ugrid_uuid:  # For testing
        return ''
    ugrid = query.item_with_uuid(ugrid_uuid)
    return ugrid.cogrid_file if ugrid else ''


def get_sim_name(query) -> str:
    """Returns the name of the simulation.

    Args:
        query (xmsapi.dmi.Query): Object for communicating with GMS

    Returns:
        (str): See description.
    """
    sim_uuid = query.parent_item_uuid()
    return tree_util.find_tree_node_by_uuid(query.project_tree, sim_uuid).name


def get_grok_filepath(query) -> Path:
    """Returns the location of the saved simulation (.grok file), and the simulation tree node.

    Args:
        query (xmsapi.dmi.Query): Object for communicating with GMS

    Returns:
        (Path): See description.
    """
    project_file = Path(XmsEnvironment.xms_environ_project_path())
    project_dir = project_file.parent
    project_base = project_file.stem
    sim_name = get_sim_name(query)
    grok_filepath = project_dir / f'{project_base}_models' / 'HydroGeoSphere' / sim_name / f'{sim_name}.grok'
    return grok_filepath


def build_solution_component(text_filepath: Path, sim_uuid: str, component_dir: str, unique_name: str):
    """Create a component for solution files.

    Args:
        text_filepath (Path): Filesystem path to the solution file.
        sim_uuid (str): The UUID of the simulation that this solution belongs to.
        component_dir (str): The component directory to add the main file to.
        unique_name (str): XML definition unique_name of the component to build

    Returns:
        data_objects.parameters.Component: The data_objects component to send back to XMS for the new component.
    """
    # create the new main file
    comp_uuid = str(uuid.uuid4())
    uuid_path = Path(component_dir) / comp_uuid
    uuid_path.mkdir(parents=True, exist_ok=True)
    main_filepath = uuid_path / f'solution.{unique_name}'
    relative_path = filesystem.compute_relative_path(str(uuid_path), str(text_filepath))
    # We store both the full path and the relative path from the component uuid dir because there are multiple
    # scenarios:
    # 1. Entire project might be moved to a new computer/location. Full path won't work, only relative will
    # 2. In save_to_location, we need the full path to compute the new relative path
    cards = {'SIM_UUID': sim_uuid, 'SOLUTION_FILE_FULL': str(text_filepath), 'SOLUTION_FILE_RELATIVE': relative_path}
    file_io_util.write_json_file(cards, main_filepath)
    do_comp = Component(
        name=text_filepath.name,
        comp_uuid=comp_uuid,
        main_file=str(main_filepath),
        model_name='HydroGeoSphere',
        unique_name=unique_name,
        locked=False
    )
    return do_comp


def find_solution_file(solution_file_full: str, solution_file_relative: str, main_file: str) -> Path | None:
    """Returns the location of the solution file or '' if not found.

    We store both the full path and the relative path from the component uuid dir because there are multiple
    scenarios:
    1. Entire project might be moved to a new computer/location. Full path won't work, only relative will
    2. In save_to_location, we need the full path to compute the new relative path

    Args:
        solution_file_full (str): Full path to the solution file.
        solution_file_relative (str): Relative path to the solution file from main_file directory.
        main_file (str): Main file of component.

    Returns:
        (str): See description.
    """
    if solution_file_full and Path(solution_file_full).is_file():
        return Path(solution_file_full)
    full_path = filesystem.resolve_relative_path(str(Path(main_file).parent), solution_file_relative)
    if full_path and Path(full_path).is_file():
        return Path(full_path)
    return None
