"""HgsComponentBase class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.components.bases.component_base import ComponentBase
from xms.guipy.resources import resources_util

# 4. Local modules
from xms.hgs.components import dmi_util


class HgsComponentBase(ComponentBase):
    """A Dynamic Model Interface (DMI) component base class."""
    def __init__(self, main_file: str) -> None:
        """Initializes the class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.filter = ''  # filter for export dialog example: ''Well Files (*.wel)'

    def save_to_location(self, new_path: str, save_type: str):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (tuple): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        # Check if we are already in the new location
        main_filepath = Path(self.main_file)
        new_main_filepath = Path(new_path) / main_filepath.name
        if new_main_filepath.resolve() == main_filepath.resolve():
            return self.main_file, [], []
        return str(new_main_filepath), [], []

    # def get_project_explorer_menus(self, main_file_list: list[str]):
    def get_project_explorer_menus(self, main_file_list: list[str]):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (list of str): A list of the main files of the selected components of this type.

        Returns:
            menu_items (list of xmsapi.dmi.MenuItem): A list of menus and menu items to be shown. Note
            that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        if not main_file_list or len(main_file_list) > 1:
            return []  # Multi-select or nothing selected

        menu_list = [None, None]  # None == spacer

        return menu_list

    def get_double_click_actions(self, lock_state: bool):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        messages: list[tuple[str]] = []
        actions = []

        open_action = ActionRequest(
            main_file=self.main_file,
            modality='modal',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='_open_dialog',
            parameters={"main_file": self.main_file}
        )
        actions.append(open_action)

        return messages, actions

    def _add_tree_menu_item(self, menu_text, method_name, menu_list, icon_file=''):
        """Convenience method for adding a menu command in the project explorer."""
        icon = resources_util.get_resource_path(f':/resources/icons/{icon_file}') if icon_file else ''
        dmi_util.add_tree_menu_item(
            menu_text=menu_text,
            method_name=method_name,
            main_file=self.main_file,
            class_name=self.class_name,
            module_name=self.module_name,
            menu_list=menu_list,
            icon=icon
        )
