"""EtpropsWriter class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.hgs.data.domains import Domains
from xms.hgs.file_io.props_writer_base import PropsWriterBase
from xms.hgs.file_io.section import IndentedSection


class EtpropsWriter(PropsWriterBase):
    """Writes the .etprops file (ET domain material properties)."""
    def __init__(self, materials_data, material_lookup, grok_filepath) -> None:
        """Initializes the class.

        Args:
            materials_data: Materials data dict.
            material_lookup: Dict of materials data arranged in a more useful way.
            grok_filepath (Path): Path to the .grok file.
        """
        super().__init__(materials_data, material_lookup, grok_filepath)
        self._domain = Domains.ET  # self._domain is defined in PropsWriterBase

    # @overrides
    def _write_material(self, i: int, material_name: str, uuid: str) -> None:
        """Writes the material."""
        heading = f'! material {i + 1}\n{material_name}'
        with IndentedSection(self._file_data, heading=heading, end_text='! material') as section:
            section.write_widgets('evaporation depth', 'edt')
            section.write_widgets('potential evaporation using transpiration')
            section.write_widgets('edf quadratic decay function')
            section.write_widgets('root depth', 'edt')
            section.write_widgets('rdf table', 'btn')
            section.write_widgets('time-root depth table', 'btn')
            section.write_widgets('rdf quadratic decay function')
            section.write_widgets('lai tables', 'btn')
            section.write_widgets('time varying lai from raster', 'btn', path_columns=[1], table=True)
            if self._materials_data[uuid]['chk_time_varying_lai_from_raster'] is True:
                section.write_value('T')  # tecplot_output
            section.write_widgets('transpiration fitting parameters', sub_widget_list=['edt_c_1', 'edt_c_2', 'edt_c_3'])
            section.write_widgets(
                'transpiration limiting pressure head',
                sub_widget_list=['edt_hwp_et', 'edt_hfc_et', 'edt_ho_et', 'edt_han_et']
            )
            section.write_widgets('evaporation limiting pressure head', sub_widget_list=['edt_he2_et', 'edt_he1_et'])
            section.write_widgets('canopy storage parameter', 'edt')
            section.write_widgets('canopy evaporation interval', 'edt')
            section.write_widgets('initial interception storage', 'edt')
