"""GrokWriter class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import logging
from pathlib import Path
import textwrap

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util, TreeNode
from xms.datasets.dataset_reader import DatasetReader
from xms.guipy.file_io_util import read_json_file, write_json_file

# 4. Local modules
from xms.hgs.components.bc_coverage_component import BcCoverageComponent
from xms.hgs.data.domains import domain_abbreviation, domain_type, Domains
from xms.hgs.file_io import file_io_util
from xms.hgs.file_io import grid_writer
from xms.hgs.file_io.etprops_writer import EtpropsWriter
from xms.hgs.file_io.file_data import FileData
from xms.hgs.file_io.mprops_writer import MpropsWriter
from xms.hgs.file_io.oprops_writer import OpropsWriter
from xms.hgs.file_io.section import FlatSection, FlatWithEndSection
from xms.hgs.gui import materials_dialog
from xms.hgs.mapping.coverage_mapper import CoverageMapper, MapAttDict

# Type aliases
MatTableTuples = list[tuple[str, str, str]]  # List of tuples of: material zone, material name, and uuid

# Constants
GMS_DATA_FILE = 'gms-data.json'


def write(main_file: str, query: Query, out_dir: Path, sim_node: TreeNode) -> Path | None:
    """Writes the .grok and all related files for the simulation and returns the .grok filepath.

    Args:
        main_file (str): Simulation component main file.
        query (Query): Object for communicating with GMS
        out_dir (Path): Path to the output directory.
        sim_node (TreeNode): Simulation tree node.
    """
    # from xms.guipy.testing import testing_tools
    # testing_tools.write_tree_to_file(query.project_tree, 'C:/temp/project_tree.txt')
    grok_writer = GrokWriter(main_file, query, out_dir, sim_node)
    return grok_writer.write()


class GrokWriter:
    """Writes all the files for the simulation.

    Grok sections are first written to a string so that the section heading is not written if section is empty. The
    sections correspond to those in the "Intro to HydroGeoSphere" tutorial
    (https://www.aquanty.com/s/HGS_Intro_Tutorial.zip).
    """

    _headings = {
        'description': '!!--------------------------  Problem description',
        'grid generation': '!!--------------------------  Grid generation',
        'simulation control': '!!--------------------------  Simulation Control Options',
        'initial conditions': '!!-------------------------- Initial Conditions',
        f'{Domains.PM} bcs': '!!-------------------------- Porous Medium BCs',
        f'{Domains.OLF} bcs': '!!-------------------------- Overland Flow BCs',
        f'{Domains.ET} bcs': '!!-------------------------- Potential Evapotranspiration BCs',
        'boundary conditions': '!!-------------------------- Boundary Conditions',
        'timestepping': '!!--------------------------  Timestep controls and Numerical Solution Criteria',
        'Porous media': '!!--------------------------  Porous media properties',
        'Surface flow': '!!--------------------------  Overland flow properties',
        'ET': '!!--------------------------  ET properties',
        'model output': '!!--------------------------  Model Output',
    }

    def __init__(self, main_file: str, query: Query, out_dir: Path, sim_node: TreeNode) -> None:
        """Initializes the class.

        Args:
            main_file: Simulation component main file.
            query: Object for communicating with GMS.
            out_dir: Path to the output directory.
            sim_node: Simulation tree node.
        """
        super().__init__()
        self._main_filepath: Path = Path(main_file)
        self._query = query
        self._out_dir = out_dir
        self._sim_node = sim_node

        self._log = logging.getLogger('xms.hgs')
        self._grok_filepath: Path = Path()
        self._temp_str: str | None = None  # Temporary string to write to instead of to the file.
        self._heading_key: str = ''  # Key (in self._headings) of the heading for the section being written
        self._first_heading = True  # Don't want a blank line before the first heading
        self._co_grid_3d = None
        self._co_grid_2d = None
        self._indent_level = 0
        self._file_data = FileData()  # Used with Section class
        self._file_data.data = read_json_file(self._main_filepath)
        self._map_atts: MapAttDict = {}
        self._set_id_strings: set[str] = set()  # Used to ensure unique grid component set id strings
        self._bc_names: set[str] = set()  # Used to ensure unique bc names
        self._bc_names_and_domains = {}

    def write(self) -> Path | None:
        """Writes the grok file and returns the grok filename."""
        if not self._set_grok_filename() or not self._find_ugrids():
            return None

        self._log.info('Writing .grok file...')
        self._write_batch_pfx()
        with open(self._grok_filepath, 'w') as self._file_data.file:
            with FlatSection(self._file_data, heading='') as _:
                self._write_description()  # Must be 1st
                self._write_grid_generation()  # Must be 2nd
                self._map_coverages_to_grid()
                self._write_simulation_control()
                self._write_all_domain_properties()
                self._write_initial_conditions()
                self._write_boundary_conditions()
                self._write_timestepping()
                self._write_model_outputs()
        self._write_gms_data()
        self._log.info('Writing finished.')
        return self._grok_filepath

    def _set_grok_filename(self) -> bool:
        """Sets up the grok filename and the ugrid uuid.

        This was too much code for __init__() so I moved it to its own function.
        """
        if not self._sim_node:
            raise RuntimeError('No simulation node found.')

        self._grok_filepath = self._out_dir / f'{self._sim_node.name}.grok'
        return True

    def _find_ugrids(self) -> bool:
        """Finds and stores the ugrid for later use."""
        self._co_grid_3d, self._co_grid_2d = file_io_util.find_and_read_ugrids(self._sim_node, self._query)
        if not self._co_grid_3d and not self._co_grid_2d:
            self._log.error('No UGrid found. A UGrid must be linked to the simulation.')
        return self._co_grid_3d is not None or self._co_grid_2d is not None

    def _write_batch_pfx(self) -> None:
        """Writes the batch.pfx file which just contains the .grok file prefix."""
        batch_pfx_filepath = self._grok_filepath.with_name('batch.pfx')
        with open(batch_pfx_filepath, 'w') as file:
            file.write(f'{self._grok_filepath.stem}\n')

    def _write_simulation_control(self) -> None:
        """Writes the general parameters."""
        self._log.info('Writing simulation control...')
        with FlatSection(self._file_data, heading=self._headings['simulation control']) as section:
            self._write_general_tab(section)
            self._write_units(section)
            self._write_saturated_flow(section)
            self._write_variably_saturated_flow(section)

    def _write_description(self) -> None:
        """Writes the description."""
        self._log.info('Writing description...')
        with FlatWithEndSection(self._file_data, heading=self._headings['description'], end_text='title') as section:
            description = self._file_data.data.get('txt_description', '')
            # Break description into lines of max length 60
            lines = textwrap.wrap(description, width=60, break_long_words=False)
            for line in lines:
                section.write_string(f'{line}')

    def _write_general_tab(self, section) -> None:
        """Writes the things in the general tab of the dialog except the description."""
        section.write_widgets('transient flow')
        section.write_widgets('unsaturated')
        section.write_widgets('finite difference mode')
        section.write_widgets('control volume')
        surface_flow = self._file_data.data.get('chk_surface_flow', False)
        if surface_flow:
            section.write_value('dual nodes for surface flow')

    def _write_units(self, section) -> None:
        """Writes the units."""
        section.write_widgets('units', one_line=True, colon=True, first_widget='cbx')  # No check box
        section.write_widgets('gravitational acceleration', 'edt')

    def _write_saturated_flow(self, section) -> None:
        """Writes the saturated flow."""
        section.write_widgets('no fluid mass balance')
        section.write_widgets('flow solver maximum iterations', 'spn')
        section.write_widgets('flow solver convergence criteria', 'edt')
        section.write_widgets('flow solver detail', 'cbx')

    def _write_variably_saturated_flow(self, section) -> None:
        """Writes the variably saturated flow."""
        section.write_widgets('upstream weighting factor', 'edt')
        section.write_widgets('remove negative coefficients')
        section.write_widgets('no nodal flow check')
        section.write_widgets('nodal flow check tolerance', 'edt')
        section.write_widgets('underrelaxation factor', 'edt')

    def _write_all_domain_properties(self) -> None:
        """Writes the properties for ALL the domains."""
        self._log.info('Writing domain properties...')
        # Get the materials dicts
        materials_data = read_json_file(self._main_filepath.with_name('materials.json'))
        material_lookup = _create_material_lookup_dict(materials_data)

        # Now write to the .grok file
        self._write_domain_properties(materials_data, material_lookup, Domains.PM)
        if self._file_data.data.get('chk_surface_flow', False):
            self._write_domain_properties(materials_data, material_lookup, Domains.OLF)
        if self._file_data.data.get('chk_et', False):
            self._write_domain_properties(materials_data, material_lookup, Domains.ET)

    def _write_domain_properties(self, materials_data: dict, material_lookup: dict, domain: str) -> None:
        """Writes the section."""
        zones_filename = self._write_zones_file(domain)  # Do this once before writing anything to grok
        if not zones_filename:
            return

        # Write the .*props file if the domain is in use, otherwise return
        props_filepath = self._write_props_file(self._grok_filepath, materials_data, material_lookup, domain)
        if not props_filepath:
            return

        self._write_domain_properties_to_grok(
            self._file_data, self._headings[domain], domain, material_lookup, props_filepath, zones_filename
        )

    @staticmethod
    def _write_domain_properties_to_grok(
        file_data, heading, domain: str, material_lookup: dict, props_filepath: Path, zones_filename: str
    ) -> None:
        """Writes the domain properties section in the .grok file.

        Static so it can be more easily tested.
        """
        if material_lookup is not None and domain in material_lookup:
            with FlatSection(file_data, heading=heading) as section:
                section.write_value('use domain type', domain_type(domain), append_blank=True)
                if props_filepath:
                    section.write_value('properties file', props_filepath.name, append_blank=True)

                    # Assign zones to elements
                    if domain == Domains.PM:
                        section.write_value('clear chosen elements')
                        section.write_value('choose elements all')
                    elif domain in {Domains.OLF, Domains.ET}:
                        section.write_value('clear chosen faces')
                        section.write_value('choose faces top')
                        section.write_value('new zone', 1)
                    else:
                        raise RuntimeError(f'Unsupported domain: "{domain}".')  # pragma no cover - too hard to get here
                    read_zones_command = GrokWriter._get_read_zones_command_for_domain(domain)
                    section.write_value(read_zones_command, zones_filename, append_blank=True)

                    # Write out each material for this domain
                    for zone, material_name, _uuid in material_lookup[domain]:
                        section.write_value('clear chosen zones')
                        section.write_value('choose zone number', zone)
                        section.write_value('read properties', material_name, append_blank=True)

    @staticmethod
    def _get_read_zones_command_for_domain(domain: str) -> str:
        """Returns the proper "read zones" command for the given domain.

        Args:
            domain (str): The domain.

        Returns:
            (str): Either "read zones from file", "read overland zones from file", or "read et zones from file".
        """
        commands = {
            Domains.PM: "read zones from file",
            Domains.OLF: "read overland zones from file",
            Domains.ET: "read et zones from file"
        }
        return commands[domain]

    @staticmethod
    def _get_zones_dataset_widgets_for_domain(domain: str) -> str:
        """Returns the proper zones widget for the given domain.

        Args:
            domain (str): The domain.

        Returns:
            (str): Either "read zones from file", "read overland zones from file", or "read et zones from file".
        """
        widgets = {
            Domains.PM: ('chk_read_porous_media_zones_from_file', 'edt_porous_media_zones_dataset'),
            Domains.OLF: ('chk_read_surface_flow_zones_from_file', 'edt_surface_flow_zones_dataset'),
            Domains.ET: ('chk_read_et_zones_from_file', 'edt_et_zones_dataset')
        }
        return widgets[domain]

    @staticmethod
    def _write_props_file(grok_filepath: Path, materials_data: dict, material_lookup: dict, domain: str) -> Path:
        """Writes the .etprops, .mprops, or .oprops file.

        Args:
            grok_filepath (Path): Path to the .grok file.
            materials_data (dict): Materials data dict.
            material_lookup (dict): Dict of materials data arranged in a more useful way.
            domain (str): The domain.

        Returns:
            (Path | None): .*props filepath.
        """
        if not materials_data or not material_lookup or domain not in material_lookup:
            return None

        mat_props_writer: MpropsWriter | OpropsWriter | EtpropsWriter
        if domain == Domains.PM:
            mat_props_writer = MpropsWriter(materials_data, material_lookup, grok_filepath)
        elif domain == Domains.OLF:
            mat_props_writer = OpropsWriter(materials_data, material_lookup, grok_filepath)
        elif domain == Domains.ET:
            mat_props_writer = EtpropsWriter(materials_data, material_lookup, grok_filepath)
        else:
            raise ValueError()  # pragma no cover - too hard to get here
        return mat_props_writer.write()

    def _write_zones_file(self, domain: str) -> str:
        """Writes the zones file and returns zone file name (just the stem, no directories)."""
        chk_widget, edt_widget = GrokWriter._get_zones_dataset_widgets_for_domain(domain)
        read_zones_from_file = self._file_data.data.get(chk_widget, False)
        if read_zones_from_file:
            zones_dataset_path = self._file_data.data.get(edt_widget)
            return GrokWriter._write_zones_file_static(
                self._query, zones_dataset_path, self._grok_filepath, domain, self._log
            )
        return ''

    @staticmethod
    def _write_zones_file_static(
        query: Query, zones_dataset_path: str, grok_filepath: Path, domain: str, logger
    ) -> str:
        """Writes the zones file and returns zone file name (just the stem, no directories).

        Static so it can be more easily tested.
        """
        if zones_dataset_path:
            dataset = GrokWriter._get_zones_dataset(query, zones_dataset_path, logger)
            if not dataset:
                return ''
            ts_data = dataset.values[0]
            zones_filepath = grok_filepath.with_name(f'{grok_filepath.stem}_{domain_abbreviation(domain)}.zon')
            with open(zones_filepath, 'w') as file:
                for index, value in enumerate(ts_data):
                    file.write(f'{index + 1} {str(int(value))}\n')
            return zones_filepath.name
        return ''

    @staticmethod
    def _get_zones_dataset(query: Query, zones_dataset_path: str, logger) -> DatasetReader:
        """Finds and returns the zones dataset.

        Args:
            query (Query): Object for communicating with GMS
            zones_dataset_path (str): Project Explorer path to the zones dataset.
            logger: Logger

        Returns:
            (DatasetReader): The dataset.
        """
        if not query:
            return None  # Only when testing
        dataset_node = tree_util.item_from_path(query.project_tree, zones_dataset_path)
        if not dataset_node and logger is not None:
            logger.error(f'Could not find dataset: {zones_dataset_path}')
            return None
        return query.item_with_uuid(dataset_node.uuid)

    def _write_initial_conditions(self):
        """Writes the initial conditions."""
        self._log.info('Writing initial conditions...')
        with FlatSection(self._file_data, heading=self._headings['initial conditions']) as section:
            # porous media
            section.write_value('use domain type', domain_type(Domains.PM), append_blank=True)
            section.write_string('choose nodes all')
            section.write_widgets('initial head', 'edt', first_widget='rbt')
            section.write_widgets('initial head depth to water table', 'edt', first_widget='rbt')
            section.write_widgets('initial head surface elevation', first_widget='rbt')
            section.write_widgets(
                'initial head from output file', sub_widget_list=['edt_initial_head_file'], first_widget='rbt'
            )

            # surface
            if self._file_data.data.get('chk_surface_flow', False):
                section.write_value('use domain type', domain_type(Domains.OLF), append_blank=True)
                section.write_string('clear chosen nodes')
                section.write_string('choose nodes all')
                section.write_widgets('initial water depth', 'edt', first_widget='rbt')
                section.write_widgets(
                    'initial water depth from file',
                    sub_widget_list=['edt_initial_water_depth_file'],
                    first_widget='rbt'
                )

    def _write_boundary_conditions(self) -> None:
        """Writes the boundary conditions."""
        self._log.info('Writing boundary conditions...')
        map_att_list = self._map_atts.get('BcCoverageComponent')
        if not map_att_list:
            return

        grouped_atts = BcCoverageComponent.group_for_grok(map_att_list)
        for grok_domain, map_atts in grouped_atts.items():
            if map_atts:
                with FlatSection(self._file_data, heading=self._headings[f'{grok_domain} bcs']) as section:
                    domain = Domains.OLF if grok_domain == Domains.ET else grok_domain
                    section.write_value('use domain type', domain_type(domain))
                    section.write_string('')
                    for map_att in map_atts:
                        map_att.write(section, self._set_id_strings, bc_names=self._bc_names)
                        self._bc_names_and_domains[map_att.value('name')] = domain_abbreviation(domain)

    def _write_timestepping(self) -> None:
        """Writes the timestepping."""
        self._log.info('Writing timestepping...')
        with FlatSection(self._file_data, heading=self._headings['timestepping']) as section:
            section.write_widgets('initial time', 'edt')
            section.write_widgets('initial timestep', 'edt')
            section.write_widgets('maximum timestep', 'edt')
            section.write_widgets('time varying maximum timestep', 'edt')
            section.write_widgets('target times', 'btn', table=True)
            section.write_widgets('minimum timestep multiplier', 'edt')
            section.write_widgets('maximum timestep multiplier', 'edt')
            section.write_widgets('jacobian epsilon', 'edt')
            section.write_widgets('minimum relaxation factor for convergence', 'edt')

            section.write_widgets('newton maximum iterations', 'spn')
            section.write_widgets('newton minimum iterations', 'spn')
            section.write_widgets('newton absolute convergence criteria', 'edt')
            section.write_widgets('newton residual convergence criteria', 'edt')
            section.write_widgets('newton maximum update for head', 'edt')
            section.write_widgets('newton maximum update for depth', 'edt')
            section.write_widgets('newton absolute maximum residual', 'edt')
            section.write_widgets('newton maximum residual increase', 'edt')

            section.write_widgets('head control', 'edt')
            section.write_widgets('water depth control', 'edt')
            section.write_widgets('saturation control', 'edt')
            section.write_widgets('newton iteration control', 'spn')

    def _write_model_outputs(self) -> None:
        """Writes the outputs."""
        self._log.info('Writing model outputs...')
        with FlatSection(self._file_data, heading=self._headings['model output']) as section:
            # Stuff we just always write that has no GUI
            section.write_value('soil water balance')
            section.write_value('generate time report')
            section.write_value('output et details')
            section.write_value('detailed runtime information')
            section.write_value('report mesh quality')
            section.write_value('track hgs output files')  # Makes prefixo.hgs_output_files.txt file
            section.write_value('track simulation progress')  # Makes prefixo.simulation_progress.csv
            # 'use legacy binary format' added in Oct 2025. HGS changed binary output file format
            section.write_value('use legacy binary format', append_blank=True)

            times = self._file_data.data.get('tbl_output_times')
            if times:
                section.write_value('output times', times, append_blank=True, table=True)

            # Write observations
            map_atts = self._map_atts.get('ObsCoverageComponent')
            if map_atts:
                for map_att in map_atts:
                    map_att.write(section=section)

            # Write hydrographs
            map_atts = self._map_atts.get('HydrographCoverageComponent')
            if map_atts:
                for map_att in map_atts:
                    map_att.write(section=section, set_id_strings=self._set_id_strings)

    def _write_grid_generation(self) -> None:
        """Writes the grid generation section."""
        self._log.info('Writing grid generation...')
        mesh_filepath = self._write_grid_generation_to_grok()
        self._write_grid(mesh_filepath)

    def _write_grid_generation_to_grok(self) -> Path | None:
        """Writes the grid generation section."""
        heading = self._headings['grid generation']
        with FlatWithEndSection(self._file_data, heading=heading, end_text='grid generation') as section:
            mesh_filepath = self._grok_filepath.with_suffix('.3dm')
            section.write_value('read GMS 3D grid', [mesh_filepath.name, 'F'], indent_list=False)
            return mesh_filepath

    def _write_grid(self, mesh_filepath) -> None:
        """Writes the grid as a .3dm file and assume it is already numbered the way HGS requires."""
        grid_writer.write(self._co_grid_3d, mesh_filepath)

    def _map_coverages_to_grid(self) -> None:
        """Intersects all the coverages with the grid."""
        self._log.info('Mapping coverages to the grid.')
        mapper = CoverageMapper(self._query, self._sim_node, self._co_grid_3d)
        self._map_atts = mapper.map()

    def _write_gms_data(self) -> None:
        """Writes the gms data file."""
        self._log.info(f'Writing {GMS_DATA_FILE} file...')
        use_start_date_time = self._file_data.data.get('chk_start_date_time', False)
        start_date_time = self._file_data.data.get('edt_start_date_time', '') if use_start_date_time else ''
        gms_data = {'bc_names_and_domains': self._bc_names_and_domains, 'start_date_time': start_date_time}
        write_json_file(gms_data, self._grok_filepath.with_name(GMS_DATA_FILE))


def _create_material_lookup_dict(materials_data: dict) -> dict | None:
    """Returns a dict of: domains -> list[(zone, name, uuid)] with the zones sorted in order."""
    if not materials_data:
        return None
    material_lookup: dict[str, MatTableTuples] = {}
    for name, domain, zones, uuid in zip(
        materials_data['table'][materials_dialog.ColumnNames.NAME],
        materials_data['table'][materials_dialog.ColumnNames.DOMAIN],
        materials_data['table'][materials_dialog.ColumnNames.ZONES],
        materials_data['table'][materials_dialog.ColumnNames.UUID]
    ):
        if domain not in material_lookup:
            material_lookup[domain] = []
        zone_list = _sorted_ints_from_string(zones)
        for zone in zone_list:
            material_lookup[domain].append((zone, name, uuid))
    return material_lookup


def _sorted_ints_from_string(zones: str) -> list:
    """Returns a sorted list of integers given a string of space separated zone numbers."""
    return sorted([int(z) for z in zones.split()])
