"""MpropsWriter class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
# from overrides import overrides

# 3. Aquaveo modules

# 4. Local modules
from xms.hgs.data.domains import Domains
from xms.hgs.file_io.props_writer_base import PropsWriterBase
from xms.hgs.file_io.section import IndentedSection


class MpropsWriter(PropsWriterBase):
    """Writes the .mprops file (porous media domain material properties)."""
    def __init__(self, materials_data: dict, material_lookup: dict, grok_filepath: Path) -> None:
        """Initializes the class.

        Args:
            materials_data (dict): Materials data dict.
            material_lookup (dict): Dict of materials data arranged in a more useful way.
            grok_filepath (Path): Path to the .grok file.
        """
        super().__init__(materials_data, material_lookup, grok_filepath)
        self._domain = Domains.PM  # self._domain is defined in PropsWriterBase

    # @overrides
    def _write_material(self, i: int, material_name: str, uuid: str) -> None:
        """Writes the material."""
        heading = f'! material {i + 1}\n{material_name}'
        with IndentedSection(self._file_data, heading=heading, end_text='! material') as section:
            if self._materials_data[uuid].get('rbt_k_isotropic', False):
                section.write_widgets('k isotropic', 'edt', first_widget='rbt')
            else:
                section.write_widgets(
                    'k anisotropic', sub_widget_list=['edt_kxx', 'edt_kyy', 'edt_kzz'], first_widget='rbt'
                )
            section.write_widgets('time dependent k for chosen elements', 'btn', item_count=True, table=True)
            section.write_widgets('specific storage', 'edt')
            section.write_widgets('porosity', 'edt')
            if self._materials_data[uuid]['cbx_unsaturated_tables_or_functions'] == 'Tables':
                with IndentedSection(
                    self._file_data, heading='unsaturated tables', end_text='! unsat tables'
                ) as tables_section:
                    tables_section.write_widgets('saturation-relative k', 'btn')
                    tables_section.write_widgets('pressure-saturation', 'btn')
            else:
                heading = self._get_unsat_functions_heading(uuid)
                with IndentedSection(
                    self._file_data, heading=heading, end_text='! unsat functions'
                ) as functions_section:
                    functions_section.write_widgets('residual saturation', 'edt')
                    functions_section.write_widgets('alpha', 'edt')
                    functions_section.write_widgets('beta', 'edt')
                    functions_section.write_widgets('pore connectivity', 'edt')
                    functions_section.write_widgets('air entry pressure', 'edt')
                    functions_section.write_widgets('exponent', 'edt')
                    functions_section.write_widgets('minimum relative permeability', 'edt')
                    functions_section.write_widgets('table smoothness factor', 'edt')
                    functions_section.write_widgets('table minimum pressure', 'edt')
                    functions_section.write_widgets('table maximum s-k slope', 'edt')
                    functions_section.write_widgets('use tabulated unsaturated functions')

    def _get_unsat_functions_heading(self, uuid: str) -> str:
        """Returns the heading for the unsaturated functions section.

        'unsaturated brooks-corey functions' or 'unsaturated van genuchten functions'

        Args:
            uuid (str): Material uuid

        Returns:
            (str): See description.
        """
        if self._materials_data[uuid]['cbx_unsaturated_functions'] == 'brooks-corey':
            heading = 'unsaturated brooks-corey functions'
        else:
            heading = 'unsaturated van genuchten functions'
        return heading
