"""TecplotFemBlockReader class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import io
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.hgs.file_io.tecplot_block_reader_base import TecplotBlockReaderBase


class TecplotFemBlockReader(TecplotBlockReaderBase):
    """Reads a Tecplot ascii fem (finite element mesh?) file and makes datasets.

    I assume the file format doesn't vary from the examples I've seen. If so, this will need to be made smarter.
    """
    def __init__(self, filepath: Path, ugrid_uuid: str, dset_time_units: str) -> None:
        """Initializes the file.

        Args:
            filepath (Path): File path.
            ugrid_uuid (str): UUID of the ugrid.
            dset_time_units (str): Dataset time units.
        """
        super().__init__(filepath, ugrid_uuid, dset_time_units)
        self._offset = 2  # To skip over x, y and z
        self._location = 'cells'
        self._title_key_words = 'TITLE',

    def _read_zone(self) -> tuple[float, list[list[float]]]:
        """Reads a ZONE block.

        Returns:
            (tuple[float, list[list[float]]): The time, and a 2d list = variable, list of values for the variable.
        """
        # if not self._line.startswith('ZONE'):
        #     raise StopIteration
        count = int(self._read_zone_string('E='))
        self._skip_blocks(self._offset)  # Skip x and y blocks

        variable_values_list = self._read_variable_values(count)
        self._file.seek(0, io.SEEK_END)  # Go to the end of the file to tell calling function we're done
        return 0.0, variable_values_list

    def _skip_blocks(self, n: int):
        """Skips n blocks.

        Args:
            n (int): Number of blocks to skip

        Returns:
            (str): The last line read (the first line after the blocks).
        """
        blocks = 0
        while True:
            if self._line.startswith('#'):
                blocks += 1
                if blocks == n + 1:
                    break
            self._line = next(self._file)
