"""TecplotOrderedBlockReader class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.hgs.file_io import file_io_util
from xms.hgs.file_io.tecplot_block_reader_base import TecplotBlockReaderBase
from xms.hgs.misc import util


class TecplotOrderedBlockReader(TecplotBlockReaderBase):
    """Reads a Tecplot ascii block file and makes datasets.

    I assume the file format doesn't vary from the examples I've seen. If so, this will need to be made smarter.
    """
    def __init__(self, filepath: Path, ugrid, ugrid_uuid: str, rtree, dset_time_units: str) -> None:
        """Initializes the file.

        Args:
            filepath (Path): File path.
            ugrid: The UGrid.
            ugrid_uuid (str): UUID of the ugrid.
            rtree: Rtree of UGrid points.
            dset_time_units (str): Dataset time units.
        """
        super().__init__(filepath, ugrid_uuid, dset_time_units)
        self._offset = 3  # To skip over x, y and z
        self._location = 'points'
        self._title_key_words = 'TITLE', 'Boundary condition'
        self._ugrid = ugrid
        self._rtree = rtree
        self._point_idxs: list[int] = []

    def _read_zone(self) -> tuple[float, list[list[float]]]:
        """Reads a ZONE block.

        Returns:
            (tuple[float, list[list[float]]): The time, and a 2d list = variable, list of values for the variable.
        """
        time, count = self._read_zone_line()

        # Read the xyz values, which only appear the first time
        if not self._point_idxs:
            x, y, z = self._read_xyz(count)
            self._find_point_indexes(x, y, z)
            self._activity = self._make_activity()

        # Read the variable values
        variable_values_list = self._read_variable_values(count)
        return time, variable_values_list

    def _read_xyz(self, count: int):
        """Reads the x, y, and z values."""
        self._line = next(self._file)
        self._line = file_io_util.skip_comments(self._file, self._line)
        x = self._read_values(count)
        self._line = next(self._file)
        self._line = file_io_util.skip_comments(self._file, self._line)
        y = self._read_values(count)
        self._line = next(self._file)
        self._line = file_io_util.skip_comments(self._file, self._line)
        z = self._read_values(count)
        return x, y, z

    def _find_point_indexes(self, xs: list[float], ys: list[float], zs: list[float]):
        """Finds the point indexes from the xyz data."""
        # Use an rtree for speed
        for x, y, z in zip(xs, ys, zs):
            nearest = list(self._rtree.nearest((x, y, z, x, y, z), 1))
            if nearest:
                self._point_idxs.append(nearest[0])

    def _read_zone_line(self) -> tuple[float, int]:
        """Reads the line starting with 'ZONE'."""
        time = float(self._read_zone_string('SOLUTIONTIME='))
        count = int(self._read_zone_string('I='))
        return time, count

    def _make_activity(self) -> list[int]:
        """Returns an activity array."""
        activity = [0] * self._ugrid.point_count
        for point_idx in self._point_idxs:
            activity[point_idx] = 1
        return activity

    def _get_timestep_values(self, variable_values: list[float]) -> list[float]:
        """Returns list of values, size of number of UGrid points, with nodata and variable_values in right spots."""
        timestep_values: list[float] = [util.nodata] * self._ugrid.point_count
        for i, point_idx in enumerate(self._point_idxs):
            timestep_values[point_idx] = variable_values[i]
        return timestep_values
