"""GuiExchange class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QCheckBox

# 3. Aquaveo modules
from xms.guipy import file_io_util
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.hgs.misc import util


def get_widget_name(type_abbreviation: str, name: str):
    """Returns the name of the widget."""
    return f'{type_abbreviation}_{name.lower().replace(" ", "_").replace("-", "_")}'


class GuiExchange:
    """Functions for exchanging data to/from gui widgets."""
    def __init__(self, ui, data, tooltip_filepath: Path) -> None:
        """Initializes the class.

        Args:
            ui: The UI class.
            data: The data dict.
            tooltip_filepath (Path): Path to the tooltip filename.
        """
        self._ui = ui
        self._data = data
        self._tooltips: dict[str, str] = {}
        self._tooltip_filepath = tooltip_filepath

    def get_data(self):
        """Returns the data dict."""
        return self._data

    def set_data(self, data):
        """Sets the data dict."""
        self._data = data

    @staticmethod
    def enable_whats_this(dialog) -> None:
        """Sets the window flags such that the "What's This" thing works in the dialog."""
        flags = dialog.windowFlags()
        flags |= Qt.WindowContextHelpButtonHint
        dialog.setWindowFlags(flags)

    def read_tooltips(self) -> None:
        """Reads the tooltip file, if it exists, and stores the tooltips."""
        self._tooltips = file_io_util.read_json_file(self._tooltip_filepath)

    def add_tooltip(self, widget, command: str) -> None:
        """Adds a tooltip to the widget if there is one."""
        tooltip = self._tooltips.get(command)
        if tooltip is not None:
            # Make it rich text so it will get word wrapped (see https://stackoverflow.com/questions/4795757)
            rich_text = f'<span>{tooltip}</span>'
            widget.setToolTip(rich_text)
            widget.setWhatsThis(rich_text)

    def widget_from_name(self, widget_name: str):
        """Returns the widget.

        Args:
            widget_name (str): String used to identify the widget (e.g. 'chk_units')
        """
        widget = getattr(self._ui, widget_name, None)
        assert widget is not None  # Shouldn't happen. There's a flaw in your naming
        return widget

    def get_widget(self, type_abbreviation: str, command: str):
        """Returns the widget given it's type abbreviation and base name."""
        return self.widget_from_name(get_widget_name(type_abbreviation, command))

    def get_widget_and_widget_name(self, type_abbreviation: str, command: str):
        """Returns the widget and its name."""
        widget_name = get_widget_name(type_abbreviation, command)
        widget = self.widget_from_name(widget_name)
        return widget, widget_name

    def get_widget_and_value(self, type_abbreviation: str, command: str, default):
        """Returns the widget and the value in the data dict."""
        widget, widget_name = self.get_widget_and_widget_name(type_abbreviation, command)
        value = self._data.get(widget_name, default)
        return widget, value

    @staticmethod
    def make_enabling_lambda(chk_or_rbt, other_widget) -> None:
        """Creates a lambda to enable/disable another widget based on the checkbox or radio button state."""
        enabler = lambda state, arg1=chk_or_rbt, arg2=other_widget: arg2.setEnabled(arg1.isChecked())  # noqa E731
        if isinstance(chk_or_rbt, QCheckBox):
            chk_or_rbt.stateChanged.connect(enabler)
        else:
            chk_or_rbt.toggled.connect(enabler)
        enabler(chk_or_rbt.isChecked())

    # ----- "do" methods which either get the data from the widgets or initialize the widgets from the data

    def do_chk(self, set_data: bool, command: str, default=False) -> None:
        """Exchanges data between the gui widgets and the data dict."""
        if set_data:
            self.data_from_chk(command)
        else:
            self.init_chk(command, default)

    def do_rbt(self, set_data: bool, command: str, default=False) -> None:
        """Exchanges data between the gui widgets and the data dict."""
        if set_data:
            self.data_from_rbt(command)
        else:
            self.init_rbt(command, default)

    def do_plain_txt(self, set_data: bool, command: str) -> None:
        """Initializes a QPlainTextEdit or gets the data from it and stores it in data dict."""
        if set_data:
            self.data_from_plain_txt(command)
        else:
            self.init_plain_txt(command)

    def do_cbx(self, set_data: bool, command: str) -> None:
        """Exchanges data between the gui widgets and the data dict."""
        if set_data:
            self.data_from_cbx(command)
        else:
            self.init_cbx(command)

    def do_chk_spn(self, set_data: bool, command: str, default) -> None:
        """Exchanges data between the gui widgets and the data dict."""
        if set_data:
            self.data_from_chk_spn(command)
        else:
            self.init_chk_spn(command, default)

    def do_chk_edt(self, set_data: bool, chk_command: str, edt_command: str, default) -> None:
        """Exchanges data between the gui widgets and the data dict."""
        if set_data:
            self.data_from_chk_edt(chk_command, edt_command)
        else:
            self.init_chk_edt(chk_command, edt_command, default)

    def do_rbt_edt(self, set_data: bool, rbt_command: str, rbt_default: bool, edt_command: str, edt_default) -> None:
        """Exchanges data between the gui widgets and the data dict."""
        if set_data:
            self.data_from_rbt_edt(rbt_command, edt_command)
        else:
            self.init_rbt_edt(rbt_command, rbt_default, edt_command, edt_default)

    def do_chk_chk(self, set_data: bool, chk1_command: str, chk2_command: str, chk2_default) -> None:
        """Exchanges data between the gui widgets and the data dict."""
        if set_data:
            self.data_from_chk_chk(chk1_command, chk2_command)
        else:
            self.init_chk_chk(chk1_command, chk2_command, chk2_default)

    def do_chk_cbx(self, set_data: bool, command: str) -> None:
        """Exchanges data between the gui widgets and the data dict."""
        if set_data:
            self.data_from_chk_cbx(command)
        else:
            self.init_chk_cbx(command)

    def do_chk_btn(self, set_data: bool, chk_command: str, btn_command: str) -> None:
        """Exchanges data between the gui widgets and the data dict."""
        if set_data:
            self.data_from_chk(chk_command)  # Buttons don't hold data so skip the button
        else:
            self.init_chk_btn(chk_command, btn_command)

    def do_rbt_btn(self, set_data: bool, rbt_command: str, btn_command: str) -> None:
        """Exchanges data between the gui widgets and the data dict."""
        if set_data:
            self.data_from_rbt(rbt_command)  # Buttons don't hold data so skip the button
        else:
            self.init_rbt_btn(rbt_command, btn_command)

    # ----- "init" methods which initialize the widgets from the data

    def init_chk(self, command: str, default: bool = False) -> QCheckBox:
        """Initializes a checkbox.

        Args:
            command (str): Widget identifier.
            default (bool): The default state of the checkbox.
        """
        widget, value = self.get_widget_and_value('chk', command, default)
        widget.setChecked(value)
        self.add_tooltip(widget, command)
        return widget

    def init_rbt(self, command: str, default: bool = False) -> QCheckBox:
        """Initializes a checkbox.

        Args:
            command (str): Widget identifier.
            default (bool): The default state of the radiobutton.
        """
        widget, value = self.get_widget_and_value('rbt', command, default)
        widget.setChecked(value)
        self.add_tooltip(widget, command)
        return widget

    def init_plain_txt(self, command: str, default: str = '') -> None:
        """Initializes a QPlainTextEdit field.

        Args:
            command (str): Widget identifier.
            default (str): Default string.
        """
        widget, value = self.get_widget_and_value('txt', command, default)
        widget.setPlainText(value)
        self.add_tooltip(widget, command)
        return widget

    def init_cbx(self, command: str):
        """Initializes a combo box from the data in the data dict.

        Args:
            command (str): Widget identifier.
        """
        widget, value = self.get_widget_and_value('cbx', command, None)
        if value is not None:
            if isinstance(value, int):
                assert value < widget.count()
                widget.setCurrentIndex(value)
            else:
                assert widget.findText(value) != -1
                widget.setCurrentText(value)
        self.add_tooltip(widget, command)
        return widget

    def init_spn(self, command: str, default):
        """Initializes a spin widget from the data in the data dict.

        Args:
            command (str): Widget identifier.
            default: Default value.
        """
        widget, value = self.get_widget_and_value('spn', command, default)
        if value is not None:
            widget.setMaximum(float(util.max_int32))
            widget.setMinimum(float(-util.max_int32))
            widget.setValue(value)
        self.add_tooltip(widget, command)
        return widget

    def init_edt(self, command: str, default):
        """Initializes an edit field from the data in the data dict.

        Args:
            command (str): Widget identifier.
            default: Default value.
        """
        widget, value = self.get_widget_and_value('edt', command, default)
        if isinstance(value, float):
            validator = QxDoubleValidator(parent=widget)
            widget.setValidator(validator)
        if value is not None:
            widget.setText(str(value))
        self.add_tooltip(widget, command)
        return widget

    def init_chk_spn(self, command: str, default) -> None:
        """Initializes a checkbox and spin widget from the data in the data dict.

        Args:
            command (str): Widget identifier.
            default: Default value for spin widget.
        """
        chk = self.init_chk(command)
        spn = self.init_spn(command, default)
        self.make_enabling_lambda(chk, spn)

    def init_chk_edt(self, chk_command: str, edt_command: str, default) -> None:
        """Initializes a checkbox and edit field from the data in the data dict.

        Args:
            chk_command (str): Widget identifier.
            edt_command (str): Widget identifier.
            default: Default value for edit field.
        """
        chk = self.init_chk(chk_command)
        edt_command = chk_command if not edt_command else edt_command
        edt = self.init_edt(edt_command, default)
        self.make_enabling_lambda(chk, edt)

    def init_rbt_edt(self, rbt_command: str, rbt_default: bool, edt_command: str, edt_default) -> None:
        """Initializes a checkbox and edit field from the data in the data dict.

        Args:
            rbt_command (str): Widget identifier.
            rbt_default: Default value for radio button.
            edt_command (str): Widget identifier.
            edt_default: Default value for edit field.
        """
        rbt = self.init_rbt(rbt_command, rbt_default)
        edt_command = rbt_command if not edt_command else edt_command
        edt = self.init_edt(edt_command, edt_default)
        self.make_enabling_lambda(rbt, edt)

    def init_chk_chk(self, chk1_command: str, chk2_command: str, chk2_default) -> None:
        """Initializes a checkbox and edit field from the data in the data dict.

        Args:
            chk1_command (str): Widget identifier.
            chk2_command (str): Widget identifier.
            chk2_default: Default value for the second checkbox.
        """
        chk1 = self.init_chk(chk1_command)
        chk2 = self.init_chk(chk2_command, chk2_default)
        self.make_enabling_lambda(chk1, chk2)

    def init_chk_cbx(self, command: str) -> None:
        """Initializes a checkbox and combo box from the data in the data dict.

        Args:
            command (str): Widget identifier.
        """
        chk = self.init_chk(command)
        cbx = self.init_cbx(command)
        self.make_enabling_lambda(chk, cbx)

    def init_chk_btn(self, chk_command: str, btn_command: str) -> None:
        """Initializes a checkbox and button from the data in the data dict."""
        chk = self.init_chk(chk_command)
        btn_command = chk_command if not btn_command else btn_command
        btn = self.widget_from_name(get_widget_name('btn', btn_command))
        self.make_enabling_lambda(chk, btn)

    def init_rbt_btn(self, rbt_command: str, btn_command: str) -> None:
        """Initializes a checkbox and button from the data in the data dict."""
        rbt = self.init_rbt(rbt_command)
        btn_command = rbt_command if not btn_command else btn_command
        btn = self.widget_from_name(get_widget_name('btn', btn_command))
        self.make_enabling_lambda(rbt, btn)

    # ----- "data_from" methods which get the data from the widgets

    def data_from_chk(self, command: str) -> None:
        """Gets the data from the widget(s) and puts it in the data dict."""
        widget, widget_name = self.get_widget_and_widget_name('chk', command)
        value = widget.isChecked()
        self._data[widget_name] = value

    def data_from_plain_txt(self, command: str) -> None:
        """Gets the data from the widget(s) and puts it in the data dict."""
        widget, widget_name = self.get_widget_and_widget_name('txt', command)
        value = widget.toPlainText()
        self._data[widget_name] = value

    def data_from_cbx(self, command: str) -> None:
        """Gets the data from the widget(s) and puts it in the data dict."""
        widget, widget_name = self.get_widget_and_widget_name('cbx', command)
        value = widget.currentText()
        self._data[widget_name] = value

    def data_from_rbt(self, command: str) -> None:
        """Gets the data from the widget(s) and puts it in the data dict."""
        widget, widget_name = self.get_widget_and_widget_name('rbt', command)
        value = widget.isChecked()
        self._data[widget_name] = value

    def data_from_edt(self, command: str) -> None:
        """Gets the data from the widget(s) and puts it in the data dict."""
        widget, widget_name = self.get_widget_and_widget_name('edt', command)
        value = widget.text()
        if widget.validator():  # Must be a float (we only expect strings or floats currently)
            value = float(value)
        self._data[widget_name] = value

    def data_from_spn(self, command: str) -> None:
        """Gets the data from the gui widgets and puts it in the data dict."""
        widget, widget_name = self.get_widget_and_widget_name('spn', command)
        value = int(widget.value())
        self._data[widget_name] = value

    def data_from_chk_edt(self, chk_command: str, edt_command: str) -> None:
        """Gets the data from the gui widgets and puts it in the data dict."""
        self.data_from_chk(chk_command)
        edt_command = chk_command if not edt_command else edt_command
        self.data_from_edt(edt_command)

    def data_from_rbt_edt(self, rbt_command: str, edt_command: str) -> None:
        """Gets the data from the gui widgets and puts it in the data dict."""
        self.data_from_rbt(rbt_command)
        edt_command = rbt_command if not edt_command else edt_command
        self.data_from_edt(edt_command)

    def data_from_chk_chk(self, chk1_command: str, chk2_command) -> None:
        """Gets the data from the gui widgets and puts it in the data dict."""
        self.data_from_chk(chk1_command)
        self.data_from_chk(chk2_command)

    def data_from_chk_spn(self, command: str) -> None:
        """Gets the data from the gui widgets and puts it in the data dict."""
        self.data_from_chk(command)
        self.data_from_spn(command)

    def data_from_chk_cbx(self, command: str) -> None:
        """Gets the data from the gui widgets and puts it in the data dict."""
        self.data_from_chk(command)
        self.data_from_cbx(command)
