"""GUI utility functions."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import os
from pathlib import Path
import subprocess

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QPalette
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.help_getter import HelpGetter
from xms.guipy.dialogs.table_dialog import TableDialog
from xms.tool_core.table_definition import TableDefinition

# 4. Local modules
from xms.hgs.misc import util


def override_icon(node):
    """Method used by TreeItemSelectorDlg to display icons.

    Args:
        node (TreeNode): A tree node.

    Returns:
        (str): Icon path.
    """
    path = ''
    if node.item_typename == 'TI_DYN_SIM_FOLDER':
        path = ':/resources/icons/hgs_simulations.svg'
    elif node.item_typename == 'TI_DYN_SIM':
        path = ':/resources/icons/hgs_simulation.svg'
    elif node.item_typename == 'TI_ROOT':
        if node.name == 'UGrid Data':
            path = ':/resources/icons/folder.svg'
    elif node.item_typename == 'TI_ROOT_SIMULATION':
        path = ':/resources/icons/simulation_data.svg'
    elif node.item_typename == 'TI_SCALAR_DSET':
        if node.name in {'Point Z', 'Cell Top Z', 'Cell Bottom Z'}:
            path = ':/resources/icons/elevation_data_inactive.svg'
        else:
            if node.data_location == 'CELL':
                path = ':/resources/icons/dataset_cells_inactive.svg'
            elif node.data_location == 'NODE':
                path = ':/resources/icons/dataset_points_inactive.svg'
    elif node.item_typename == 'TI_SOLUTION_FOLDER':
        path = ':/resources/icons/folder_locked.svg'
    # TreeNode doesn't have type of UGrid constraint so we can't do any better yet.
    # elif node.item_typename == 'TI_UGRID':
    #     path = ''
    return path


def set_read_only_and_grey(widget, on):
    """Sets the widget to be read-only and changes the color to grey, or the reverse.

    Args:
        widget: The widget.
        on (bool): Whether we are setting or unsetting.
    """
    widget.setReadOnly(on)
    if on:
        read_only_palette = QPalette()
        read_only_palette.setColor(QPalette.Base, widget.palette().color(QPalette.Window))
        widget.setPalette(read_only_palette)
    else:
        my_type = type(widget)
        default_widget = my_type()
        widget.setPalette(default_widget.palette())


def setup_table_context_menus(table, header_column_method, general_method):
    """Sets up context menus for the header column and everywhere else in the table.

    Args:
        table: The table. Something derived from QTableView.
        header_column_method: Method to call when right-clicking in the header column.
        general_method: Method to call when right-clicking anywhere else.
    """
    table.verticalHeader().setContextMenuPolicy(Qt.CustomContextMenu)
    table.verticalHeader().customContextMenuRequested.connect(header_column_method)
    table.setContextMenuPolicy(Qt.CustomContextMenu)
    table.customContextMenuRequested.connect(general_method)


def get_unique_sorted_selected_rows(selected_rows):
    """Returns a sorted list of unique selected rows.

     Selection may be disjoint.

    Args:
        selected_rows (list of QModelIndex): The selected list.

    Returns:
        (list[int]): See description.
    """
    return sorted(list({index.row() for index in selected_rows}))


def rows_are_contiguous(unique_selected_rows):
    """Returns true if the selected rows are not contiguous."""
    if not unique_selected_rows:
        return False
    return len(unique_selected_rows) == max(unique_selected_rows) - min(unique_selected_rows) + 1


def get_wiki_help_url() -> str:
    """Returns something like 'https://www.xmswiki.com/wiki/GMS:GMS_10.5_Dialog_Help'."""
    xms_name = util.get_xms_name()
    if xms_name is None:
        xms_name = 'GMS'
    xms_version = util.get_xms_version()
    if xms_version is None:
        xms_version = '99.99'
    return f'https://www.xmswiki.com/wiki/{xms_name}:{xms_name}_{xms_version}_Dialog_Help'


def help_getter(key: str) -> HelpGetter:
    """Return a HelpUrl set up using DialogInput.

    Args:
        key: The second part of the wiki help line on the above page (after the '|').

    Returns:
        See description.
    """
    default = 'https://www.xmswiki.com/wiki/GMS:HydroGeoSphere'
    return HelpGetter(key=key, default=default, dialog_help_url=get_wiki_help_url())


def run_table_dialog(title: str, table_definition: TableDefinition, json_str: str, parent, button: str) -> str | None:
    """Runs the TableDialog.

    Args:
        title (str): Window title.
        table_definition (TableDefinition): Defines the table (column types, fixed row count or not).
        json_str: The table as a json string (see DataFrame.to_json(), pandas.read_json(orient='table')).
        parent (QObject): The dialog's Qt parent.
        button (str): Name of the button widget.

    Returns:
        None on Cancel, or pandas string (see DataFrame.to_json(orient='table'), pandas.read_json(orient='table')).
    """
    df = table_definition.to_pandas(json_str)
    dialog = TableDialog(title, table_definition, df, parent=parent, dlg_name=f'xms.hgs.gui.table_dialog.{button}')
    if dialog.exec() == QDialog.Accepted:
        df = dialog.get_values()
        return df.to_json(orient='table')
    return None


def open_file_in_default_app_or_notepad(filepath: Path, parent_window) -> None:
    """Opens the file in the default app or, if none, Notepad.

    Args:
        filepath (Path): The file path.
        parent_window: The parent window.
    """
    try:
        os.startfile(filepath, 'open')
    except OSError:
        try:
            subprocess.Popen(['notepad', str(filepath)])
        except OSError:
            message = f'Could not open file "{str(filepath)}". Check for app associated with "{filepath.suffix}" files.'
            message_box.message_with_ok(parent=parent_window, message=message, app_name='GMS')
