"""Code to make node/face/segment sets."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.hgs.file_io import file_io_util
from xms.hgs.misc import util


def make_set(grok_filepath: Path, map_att, set_id_strings: set[str], section) -> tuple[str, str]:
    """Makes the set.

    Args:
        grok_filepath (Path): Path to the .grok file.
        map_att: The MapAtt.
        set_id_strings (set[str]): Set of grid component set id strings so we can ensure they are unique.
        section: The section we are writing to.

    Returns:
        (tuple[str, str]): Set name and set id str.
    """
    # We use a class so we don't have to pass so many arguments around
    maker = SetMaker(grok_filepath, map_att, set_id_strings, section)
    return maker.make_set()


class SetMaker:
    """Makes sets."""
    def __init__(self, grok_filepath: Path, map_att, set_id_strings: set[str], section) -> None:
        """Initializes the class.

        Args:
            grok_filepath (Path): Path to the .grok file.
            map_att: The MapAtt.
            set_id_strings (set[str]): Set of grid component set id strings so we can ensure they are unique.
            section: The section we are writing to.
        """
        self._grok_filepath = grok_filepath
        self._map_att = map_att
        self._set_id_strings = set_id_strings
        self._section = section

    def make_set(self) -> tuple[str, str]:
        """Makes the set.

        Returns:
            (tuple[str, str]): Set name and set id str.
        """
        set_id_str = self._make_set_id_str()
        grid_component_type = self._map_att.get_grid_component_type()
        if grid_component_type == 'node':
            set_name = self._create_node_or_element_set(set_id_str, 'node', self._map_att.intersection['points'])
        # elif grid_component_type == 'element':
        #     set_name = self._create_node_or_element_set(set_id_str, 'element', self._map_att.intersection['cells'])
        elif grid_component_type == 'face':
            set_name = self._create_face_set(set_id_str)
        elif grid_component_type == 'segment':
            set_name = self._create_segment_set(set_id_str, self._map_att.intersection['points'])
        else:
            raise RuntimeError(f'Unsupported set "{grid_component_type}"')  # pragma no cover - cost/benefit too high

        return set_name, set_id_str

    def _make_set_id_str(self) -> str:
        """Returns a short string that uniquely identifies the set.

        Use either the name the user gave the bc, or the bc type.

        Returns:
            (str): See description.
        """
        name = self._map_att.value('name')
        if not name:
            name = self._map_att.att_type.lower()

        name = name.replace(' ', '-')  # Replace spaces with dashes
        return file_io_util.make_string_unique(name, self._set_id_strings)

    def _create_node_or_element_set(self, set_id_str: str, set_type: str, chosen_items) -> str:
        """Creates a node/element/face set and returns the name of the file used by the 'choose nodes list' command.

        Args:
            set_id_str (str): String that uniquely identifies the bc.
            set_type (str): 'node', 'element' etc. Singular.
            chosen_items: Point or cell indices.

        Returns:
            (str): Name of the set.
        """
        self._section.write_string(f'clear chosen {set_type}s')
        chosen_items = [i + 1 for i in chosen_items]  # Make 1-based
        set_list_filepath = self._write_chosen_items_to_file(set_id_str, set_type, chosen_items)
        self._section.write_value(f'choose {set_type}s list', set_list_filepath.name)
        self._section.write_value(f'create {set_type} set', set_list_filepath.stem)
        self._section.write_string('')
        return set_list_filepath.stem

    def _create_face_set(self, set_id_str: str) -> str:
        """Creates a face set and returns the name of the file used by the 'Choose faces by nodes list' command.

        Args:
            set_id_str (str): String that uniquely identifies the bc.

        Returns:
            (str): Name of the set.
        """
        face_nodes = self._map_att.intersection['face nodes']
        self._section.write_string('clear chosen faces')
        set_list_filepath = self._write_chosen_items_to_file(set_id_str, 'face', face_nodes)
        self._section.write_value('choose faces by nodes list')
        self._section.write_value('include', set_list_filepath.name, one_line=True)
        self._section.write_value('end', append_blank=True)
        self._section.write_value('create face set from chosen faces', set_list_filepath.stem, append_blank=True)
        return set_list_filepath.stem

    def _create_segment_set(self, set_id_str: str, chosen_items) -> str:
        """Creates a segment set and returns its name.

        Args:
            set_id_str (str): String that uniquely identifies the bc.
            chosen_items: Point indices.

        Returns:
            (str): Name of the set.
        """
        self._section.write_string('clear chosen segments')
        self._section.write_string('clear chosen nodes')
        chosen_items = [i + 1 for i in chosen_items]  # Make 1-based
        set_list_filepath = self._write_chosen_items_to_file(set_id_str, 'node', chosen_items)
        self._section.write_value('choose nodes list', set_list_filepath.name)
        self._section.write_value('create segment set', set_list_filepath.stem)
        self._section.write_string('')
        return set_list_filepath.stem

    def _write_chosen_items_to_file(self, set_id_str: str, set_type: str, chosen_set_items) -> Path:
        """Writes the file used with the 'Choose node list' or 'Choose elements list' commands.

        Args:
            set_id_str (str): String that uniquely identifies the set.
            set_type (str): 'node', 'element' etc. Singular.
            chosen_set_items: Point indices.

        Returns:
            (Path): Path of the file written.
        """
        filepath = self._filepath_of_set(self._grok_filepath, set_id_str, set_type)
        with filepath.open('w') as file:
            for item in chosen_set_items:
                if util.is_non_string_sequence(item):  # table
                    file.write(f'{" ".join(map(str, item))}\n')
                else:
                    file.write(f'{item}\n')
        return filepath

    @staticmethod
    def _filepath_of_set(grok_filepath: Path, set_id_str: str, set_type: str) -> Path:
        """Returns the filepath of the node, face, segment set file.

        Args:
            grok_filepath (Path): Path to the .grok file.
            set_id_str (str): String that uniquely identifies the bc.
            set_type (str): 'node', 'element' etc. Singular.

        Returns:
            (Path): The filepath.
        """
        return grok_filepath.with_stem(f'{set_id_str}-{set_type}s').with_suffix('.txt')
