"""Settings class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy import file_io_util

# 4. Local modules


class Settings:
    """Class to handle settings files, which are json files saved next to a main_file."""
    def __init__(self) -> None:
        """Initializes the class."""

    @classmethod
    def settings_filepath(cls, main_filepath: Path, prefix: str = 'settings') -> Path:
        """Returns the filepath to the settings file.

        Args:
            main_filepath (Path): File path of simulation main file.
            prefix (str): Prefix part of settings file name.

        Returns:
            See description.
        """
        return main_filepath.with_name(f'{prefix}.json') if main_filepath else Path()

    @classmethod
    def read_settings(cls, main_filepath: Path, prefix: str = 'settings'):
        """Returns settings dict from disk or the default if not found.

        Args:
            main_filepath (Path): File path of simulation main file.
            prefix (str): Prefix part of settings file name.

        Returns:
            (dict[str, obj]): See description.
        """
        settings_filepath = Settings.settings_filepath(main_filepath, prefix)
        settings = file_io_util.read_json_file(settings_filepath)
        return settings

    @classmethod
    def write_settings(cls, main_filepath: Path, settings, prefix: str = 'settings') -> None:
        """Writes the settings dict to disk.

        Args:
            main_filepath (Path): File path of simulation main file.
            settings (dict[str, obj]): Dict of settings.
            prefix (str): Prefix part of settings file name.
        """
        settings_filepath = Settings.settings_filepath(main_filepath, prefix)
        file_io_util.write_json_file(settings, settings_filepath)

    @classmethod
    def get(cls, main_filepath: Path, variable: str, default=None, prefix: str = 'settings'):
        """Returns the variable or None if not found.

        Args:
            main_filepath (Path): File path of simulation main file.
            variable (str): Name of the variable to get the value of.
            default: Default value to return if the variable is not found.
            prefix (str): Prefix part of settings file name.

        Returns:
            See description.
        """
        settings = cls.read_settings(main_filepath, prefix)
        return settings.get(variable, default)

    @classmethod
    def set(cls, main_filepath: Path, variable: str, value, prefix='settings') -> None:
        """Sets the last used directory.

        Args:
            main_filepath (Path): File path of simulation main file.
            variable (str): Name of the variable to get the value of.
            value: The value of the variable to set.
            prefix (str): Prefix part of settings file name.
        """
        settings = cls.read_settings(main_filepath, prefix)
        settings[variable] = value
        cls.write_settings(main_filepath, settings, prefix)
