"""Code to create a .3dm file from a tecplot mesh file."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import argparse
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ThreeDMeshFromTecplotGenerator:
    """Creates a .3dm file from a tecplot mesh file."""
    def __init__(self):
        """Initializes the class."""
        self._tecplot_filepath = None
        self._tecplot_file = None
        self._node_count = 0
        self._element_count = 0
        self._x = []
        self._y = []
        self._z = []
        self._elements = []
        self._element_type = ''

    def generate(self, tecplot_filepath: Path) -> Path | None:
        """Creates a .3dm file from a tecplot mesh file.

        The .3dm file will have the same file name but with a .3dm extension.

        Args:
            tecplot_filepath (Path): File path of tecplot mesh file.

        Returns:
            (Path): File path of .3dm file, or None if errors.
        """
        try:
            self._tecplot_filepath = tecplot_filepath
            with self._tecplot_filepath.open('r') as self._tecplot_file:
                self._read_dimensions()
                self._read_xyz('# x', self._x)
                self._read_xyz('# y', self._y)
                self._read_xyz('# z', self._z)
                self._read_elements()
            return self._write_3dm()
        except Exception as ex:
            print(ex)
            return None

    def _advance_to(self, starts_with_string: str) -> str | None:
        """Advances the file to the line that starts with starts_with_string.

        Args:
            starts_with_string (str): The first part of the line that we're looking for.

        Returns:
            (str): The line, if found, else None.
        """
        for line in self._tecplot_file:
            if line.startswith(starts_with_string):
                return line
        return None

    def _read_dimensions(self) -> None:
        """Reads the number of nodes and elements."""
        line = self._advance_to('ZONE')
        words = [word.strip() for word in line.split(',')]  # split on comma and strip spaces
        for word in words:
            if word.startswith('N='):
                self._node_count = int(word[2:])
            elif word.startswith('E='):
                self._element_count = int(word[2:])

    def _read_xyz(self, starts_with_string, the_list):
        """Reads the x, y, or z coordinates."""
        self._advance_to(starts_with_string)
        for line in self._tecplot_file:
            words = line.split()
            for word in words:
                the_list.append(float(word))
            if len(the_list) == self._node_count:
                break

    def _determine_element_type(self) -> tuple[str, str]:
        """Returns either 'prism' or 'hexahedron', and the line from the file.

        Returns:
            (str, str): See description.
        """
        line = self._tecplot_file.readline()
        words = line.split()
        if words[2] == words[3] and words[6] == words[7]:
            return 'prism', line
        else:
            return 'hexahedron', line

    def _read_elements(self):
        """Reads the elements."""
        self._advance_to('# element node lists')
        self._element_type, line = self._determine_element_type()
        if self._element_type == 'prism':
            self._read_element_lines(line, self._read_prism)
        else:
            self._read_element_lines(line, self._read_hexahedron)

    def _read_element_lines(self, line: str, method) -> None:
        """Reads the elements.

        Args:
            line (str): The first line from the elements list.
            method: The method with which to read the element (either prism or hexahedron).
        """
        # Do the first one outside the for loop because we already read the line
        method(line)
        count = 1
        if count == self._element_count:
            return

        for line in self._tecplot_file:
            method(line)
            count += 1
            if count == self._element_count:
                return

    def _read_prism(self, line):
        """Reads a prism element from a line."""
        words = line.split()
        element = [*list(map(int, words[:2])), *list(map(int, words[3:-1]))]
        self._elements.append(element)

    def _read_hexahedron(self, line):
        """Reads a prism element from a line."""
        words = line.split()
        element = list(map(int, words))
        self._elements.append(element)

    def _write_3dm(self) -> Path:
        """Writes the .3dm file.

        Returns:
            (str): File path of .3dm file.
        """
        _3dm_filepath = self._tecplot_filepath.with_suffix('.3dm')
        with _3dm_filepath.open('w') as _3dm_file:
            _3dm_file.write('MESH3D\n')
            self._write_elements(_3dm_file)
            self._write_nodes(_3dm_file)
        return _3dm_filepath

    def _write_elements(self, _3dm_file) -> None:
        """Writes the elements.

        Args:
            _3dm_file: The .3dm file.
        """
        # GMS format is bottom, then top, counterclockwise looking down
        if self._element_type == 'prism':
            for i, el in enumerate(self._elements):
                _3dm_file.write(f'E6W {i + 1} {el[0]} {el[1]} {el[2]} {el[3]} {el[4]} {el[5]} 1\n')
        else:
            for i, el in enumerate(self._elements):
                _3dm_file.write(f'E8H {i + 1} {el[0]} {el[1]} {el[2]} {el[3]} {el[4]} {el[5]} {el[6]} {el[7]} 1\n')

    def _write_nodes(self, _3dm_file):
        """Writes the nodes.

        Args:
            _3dm_file: The .3dm file.
        """
        for i in range(self._node_count):
            _3dm_file.write(f'ND {i + 1} {self._x[i]} {self._y[i]} {self._z[i]}\n')


def main(tecplot_mesh_file):
    """Runs the script."""
    generator = ThreeDMeshFromTecplotGenerator()
    threedm_path = generator.generate(tecplot_mesh_file)
    if threedm_path:
        print(f'File created: {str(threedm_path)}')
        return 0
    else:
        print('Errors - .3dm file not generated.')
        return 1


if __name__ == "__main__":  # pragma no cover - can't run this from the tests and it must be here
    parser = argparse.ArgumentParser()
    parser.add_argument('tecplot_mesh_file')
    args = parser.parse_args()
    main(Path(args.tecplot_mesh_file))
