"""SimRunner class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import logging
import os
from pathlib import Path
import subprocess

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query, XmsEnvironment
from xms.api.tree import tree_util
from xms.components.bases.run_base import RunBase
from xms.coverage.xy import xy_io
from xms.guipy.dialogs import message_box
from xms.guipy.settings import SettingsManager

# 4. Local modules
from xms.hgs.components import dmi_util
from xms.hgs.file_io import file_io_util
from xms.hgs.file_io import solution_reader

GROK_EXE = 'HydroGeoSphere - HydroGeoSphere - Grok'
PHGS_EXE = 'HydroGeoSphere - HydroGeoSphere - Phgs'
HSPLOT_EXE = 'HydroGeoSphere - HydroGeoSphere - Hsplot'


def grok_file_path_from_sim_node_and_dir(sim_name: str, sim_dir: str) -> Path:
    """Returns the file path to the .grok file.

    Args:
        sim_name (str): The Simulation you want to load the solution for.
        sim_dir (str): The location of input files for the simulation.
    """
    return Path(sim_dir) / f'{sim_name}.grok'


def find_grok_file_or_warn(sim=None, sim_dir: str = '', grok_filepath: Path = None, ask_if_saved: bool = True) -> bool:
    """Checks for existence of the .grok file and returns True if found, or warns user and returns False.

    Either specify sim and sim_dir, or grok_file_path, but not both.

    Args:
        sim (data_objects.parameters.Simulation): The Simulation you want to load the solution for.
        sim_dir (str): The location of input files for the simulation.
        grok_filepath (Path): Path to the .grok file.
        ask_if_saved (bool): If True, asks the user if they've saved the simulation if it isn't found.
    """
    # Check that arguments are used properly
    args_passed = (bool(sim), bool(sim_dir), bool(grok_filepath))
    if args_passed != (True, True, False) and args_passed != (False, False, True):
        logger = logging.getLogger('xms.hgs')
        logger.debug('find_grok_file_or_warn(): Either specify sim and sim_dir, or grok_filepath.')
        return False

    if sim and sim_dir:
        grok_filepath = grok_file_path_from_sim_node_and_dir(sim.name, sim_dir)
    if not grok_filepath.is_file():
        message = f'No simulation found at "{str(grok_filepath)}".'  # noqa: B028
        if ask_if_saved:
            message += ' Have you saved the simulation?'
        message_box.message_with_ok(parent=None, message=message, app_name='GMS')
        return False
    return True


def get_gms_hgs_executable_paths() -> dict[str, Path]:
    """Returns the path to the HydroGeoSphere executables as specified by the GMS preferences.

    If testing, and we can't get the path from the GMS preferences, we look for the installed executables.

    Returns:
        (dict[str, str]): See description.
    """
    exes: dict[str, Path] = {}
    settings = SettingsManager(python_path=False)
    package = 'Model Executables 64 bit'
    _add_exe_path(settings, package, GROK_EXE, 'grok.exe', exes)
    _add_exe_path(settings, package, PHGS_EXE, 'phgs.exe', exes)
    _add_exe_path(settings, package, HSPLOT_EXE, 'hsplot.exe', exes)
    return exes


def get_gms_hgs_executable(exe_key: str) -> Path:
    """Returns the path to the HydroGeoSphere executables as specified by the GMS preferences.

    If testing, and we can't get the path from the GMS preferences, we look for the installed executables.

    Returns:
        (dict[str, str]): See description.
    """
    return get_gms_hgs_executable_paths().get(exe_key)


def run_hgs_exe(exe_key: str, grok_filepath: Path, console: bool = False, stdout=None):
    """Runs grok.exe or phgs.exe or hsplot.exe in a separate process.

    Returns immediately after starting the exe, returning the process Popen instance.

    Args:
        exe_key (str): Key used to identify the executable (GROK_EXE, PHGS_EXE, or HSPLOT_EXE).
        grok_filepath (Path): Path to the .grok file.
        console (bool): If True, the console window is shown.
        stdout: What stdout gets written to.

    Returns:
        Whatever subprocess.run() or subprocess.Popen() returns.
    """
    if not find_grok_file_or_warn(grok_filepath=grok_filepath):
        return None

    exe_path = get_gms_hgs_executable_paths()[exe_key]
    if not exe_path.is_file():
        message = f'No executable found at "{str(exe_path)}".'  # noqa: B028
        message_box.message_with_ok(parent=None, message=message, app_name='GMS')
        return None

    # Launch process, return immediately, but keep the console window open
    # # See https://stackoverflow.com/questions/89228/
    # # See https://stackoverflow.com/questions/11615455/
    # # /wait = subprocess.run() won't return until process is done (console window becomes "modal")
    # # /K = makes the console window stay open after the hgs executable finishes
    # # args = ['start', '/wait', 'cmd', '/K', str(exe_path)]
    # args = ['start', 'cmd', '/K', str(exe_path)]
    # rv = subprocess.run(args=args, cwd=grok_filepath.parent, shell=True)

    flags = subprocess.CREATE_NEW_CONSOLE if console else 0
    rv = subprocess.Popen(
        exe_path, cwd=str(grok_filepath.parent), creationflags=flags, stdout=stdout, stderr=subprocess.STDOUT
    )
    return rv


class SimRunner(RunBase):
    """Class that handles running HydroGeoSphere."""
    def __init__(self, dummy_mainfile: str = '') -> None:
        """Initializes the class.

        Args:
            dummy_mainfile (str): Unused. Just to keep constructor consistent with component classes.
        """
        super().__init__()

    def read_solution(self, query: Query, params, win_cont):
        """Reads the HydroGeoSphere Solution.

        Args:
            query (data_objects.parameters.Query): a Query object to communicate with GMS.
            params (dict): Generic map of parameters. Contains the structures for various components that
             are required for adding vertices to the Query Context with Add().
            win_cont (QWidget): Parent window

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        sim_uuid = query.parent_item_uuid()
        components_dir = os.path.join(query.xms_temp_directory, 'Components')
        grok_filepath = Path(params[0]['grok_file'])
        sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, sim_uuid)
        co_grid_3d, co_grid_2d = file_io_util.find_and_read_ugrids(sim_node, query)
        datasets_and_folders, xy_series_list = solution_reader.run(grok_filepath, co_grid_3d, co_grid_2d, win_cont)

        # Add the datasets
        for dataset_writer, folder in datasets_and_folders:
            query.add_dataset(dataset_writer, folder_path=folder)

        # Save the XySeries to disk
        xy_series_filepath = grok_filepath.with_name('xy_series.h5')
        xy_io.write_to_h5(xy_series_filepath, xy_series_list)

        # Add .lst file
        self._add_list_file(grok_filepath, sim_uuid, components_dir, query)
        return [], []

    def _add_list_file(self, grok_filepath: Path, sim_uuid: str, components_dir: str, query):
        """Adds the .lst file."""
        list_filepath = grok_filepath.with_name(f'{grok_filepath.stem}o.lst')
        if list_filepath.is_file():
            txt_comp = dmi_util.build_solution_component(list_filepath, sim_uuid, components_dir, 'TXT_SOL')
            query.add_component(txt_comp)

    def get_executables(self, sim, query: Query, filelocation: str) -> list:
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (data_objects.parameters.Simulation): The Simulation you want to load the solution for.
            query (data_objects.parameters.Query): a Query object to communicate with GMS.
            filelocation (str): The location of input files for the simulation.

        Returns:
            (list[ExecutableCommand]): The executable objects to run and the action requests that go with it.
        """
        return []

    def get_solution_load_actions(self, sim, query: Query, filelocation: str) -> list:
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (data_objects.parameters.Simulation): The Simulation you want to load the solution for.
            query (data_objects.parameters.Query): a Query object to communicate with GMS.
            filelocation (str): The location of input files for the simulation.

        Returns:
            (list[ActionRequest]): The action requests.
        """
        if not find_grok_file_or_warn(sim=sim, sim_dir=filelocation, ask_if_saved=False):
            return []

        grok_filepath = grok_file_path_from_sim_node_and_dir(sim.name, filelocation)
        load_sol = ActionRequest(
            main_file=str(grok_filepath),
            modality='MODAL',
            class_name='SimRunner',
            module_name='xms.hgs.simulation_runner.sim_runner',
            method_name='read_solution',
            parameters={'grok_file': str(grok_filepath)}
        )
        return [load_sol]


def _get_installed_exe_path(exe_filename: str) -> str:
    """Searches for the installed HydroGeoSphere executable.

    Args:
        exe_filename (str): Name of the HydroGeoSphere executable.

    Returns:
        (str): The path to the executable.
    """
    # Import the hgs executables here so that if we forget to install it, we're more likely to figure that out
    paths = [
        'C:/Program_Files/HydroGeoSphere/grok.exe', 'C:/Program_Files/HydroGeoSphere/phgs.exe',
        'C:/Program_Files/HydroGeoSphere/hsplot.exe'
    ]
    for path in paths:
        if path.lower().endswith(exe_filename):
            return path
    return ''


def _add_exe_path(
    settings_manager: SettingsManager, package: str, exe_key: str, exe_name: str, exes: dict[str, Path]
) -> None:
    """Adds an item to the exes dict.

    Args:
        settings_manager (SettingsManager): The guipy SettingsManager.
        package (str): The registry folder?
        exe_key (str): Key used to identify the executable (GROK_EXE, PHGS_EXE, or HSPLOT_EXE).
        exe_name (str): Executable file name.
        exes (dict[str, str]): Dict of executables and their paths.
    """
    val = settings_manager.get_setting(package, exe_key, '')
    if (not val or val == 'NONE') and XmsEnvironment.xms_environ_running_tests() == 'TRUE':
        val = _get_installed_exe_path(exe_name)
    exes[exe_key] = Path(os.path.abspath(val))
