"""Module for the find_template function."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment
from xms.api.tree import tree_util
from xms.guipy.dialogs.file_selector_dialogs import get_open_filename
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.process_feedback_dlg import run_feedback_dialog

# 4. Local modules
from xms.hydroas.data.coverage_data import CoverageData
from xms.hydroas.data.data import get_version
from xms.hydroas.data.sim_data import SimData
from xms.hydroas.feedback.read_template_runner import ReadTemplateRunner
from xms.hydroas.file_io.gmi_reader import GmiReader
from xms.hydroas.resources import default_template_path


def find_template(created_data: CoverageData | SimData, query: Query, parent: QWidget):
    """
    Find a template for a newly created HydroAS component.

    HydroAS components are largely useless until their template (stored in the generic_model attribute) is initialized.
    The reader code initializes the template using the contents of the .2dm file, but components created by the user
    through the UI can't use that. This function initializes the component's template by asking the user for either a
    .2dm file or another tree item to use as a reference, then copying its template into the newly created component.

    It is possible for this function to fail, e.g. because the user cancelled. If this happens, the newly created
    component will be deleted.

    Args:
        created_data: Data manager for the newly created component.
        query: Interprocess communication object.
        parent: Parent widget for dialogs.

    Returns:
        Messages and ActionRequests for a component event handler to return.
    """
    success, values = _use_user_selected_template(created_data, parent)
    if not success:
        values = _use_generic_template(created_data)
    _rename_item(created_data, values, query)


def _use_user_selected_template(created_data: CoverageData | SimData, parent: QWidget) -> tuple[bool, str]:
    """
    Try to use a user-selected template.

    Returns:
        Whether it was successful and, if so, the model values.
    """
    file_path = get_open_filename(parent, 'Choose .2dm template file', '.2dm template files (*.2dm)')
    if not file_path:
        message_with_ok(
            parent, 'No template was provided. A default one will be used.', XmsEnvironment.xms_environ_app_name()
        )
        return False, ''

    worker = ReadTemplateRunner(file_path)
    run_feedback_dialog(worker, parent)
    if worker.model is None:
        message_with_ok(
            parent, 'The template was invalid. A default one will be used.', XmsEnvironment.xms_environ_app_name()
        )
        return False, ''

    created_data.generic_model = worker.model
    if isinstance(created_data, SimData):
        created_data.model_values = worker.model_values
    created_data.commit()

    return True, worker.model_values


def _use_generic_template(created_data: CoverageData | SimData):
    """Add the generic template to the component."""
    reader = GmiReader(default_template_path)
    reader.read()

    created_data.generic_model = reader.model
    if isinstance(created_data, SimData):
        created_data.model_values = reader.model_instantiation
    created_data.commit()

    return reader.model_instantiation


def _rename_item(created_data: CoverageData | SimData, values: str, query: Query):
    """Rename the item to add its version."""
    model = created_data.generic_model
    version = get_version(model, values)

    tree = query.copy_project_tree()
    item_node = tree_util.find_tree_node_by_uuid(tree, query.parent_item_uuid())
    current_name: str = item_node.name

    new_name = f'{current_name} [{version}]'
    query.rename_item(query.parent_item_uuid(), new_name)
