"""Module for MappedMaterialData."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['MappedMaterialData']

# 1. Standard Python modules
from contextlib import suppress
import os
from pathlib import Path
from typing import Optional, Sequence

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.gmi.data.generic_model import GenericModel

# 4. Local modules


def make_material_dataset(cells: Optional[Sequence[int]] = None) -> xr.Dataset:
    """
    Make a dataset for the mapping from cell index to material ID.

    Args:
        cells: Sequence of material IDs. Parallel to the cell list for the geometry.

    Returns:
        A dataset for the mapping.
    """
    cells = cells if cells is not None else []
    dataset = {'cells': np.array(cells, dtype=np.int32)}
    return xr.Dataset(data_vars=dataset)


class MappedMaterialData(XarrayBase):
    """Data manager for a MappedMaterialComponent."""
    def __init__(self, main_file: str | Path):
        """
        Initialize the class.

        Args:
            main_file: The data manager's main-file.
        """
        super().__init__(main_file)
        self._cell_materials: Optional[xr.Dataset] = None
        self.uuid = os.path.basename(os.path.dirname(main_file))
        self.main_file = str(main_file)

    @property
    def material_values(self) -> str:
        """
        The shared values for material parameters.

        This does not include group activity.
        """
        if 'MATERIAL_VALUES' in self.info.attrs:
            return self.info.attrs['MATERIAL_VALUES']
        # Let GenericModel define the defaults in case it wants to change them.
        return GenericModel().material_parameters.extract_values()

    @material_values.setter
    def material_values(self, value: str):
        """
        The shared values for material parameters.

        This does not include group activity.
        """
        self.info.attrs['MATERIAL_VALUES'] = value

    @property
    def generic_model(self) -> GenericModel:
        """The model definition used to create the mapped materials."""
        if 'GENERIC_MODEL_TEMPLATE' in self.info.attrs:
            definition = self.info.attrs['GENERIC_MODEL_TEMPLATE']
            model = GenericModel(definitions=definition)
            return model
        return GenericModel()

    @generic_model.setter
    def generic_model(self, model: GenericModel):
        """The model definition used to create the mapped materials."""
        self.info.attrs['GENERIC_MODEL_TEMPLATE'] = model.to_template().to_string()

    @property
    def cell_materials(self) -> Sequence[int]:
        """Material indexes for each cell."""
        if self._cell_materials is None:
            with suppress(FileNotFoundError, OSError):
                self._cell_materials = xr.load_dataset(self._filename, group='cell_materials')

        if self._cell_materials is None:
            self._cell_materials = make_material_dataset()

        return self._cell_materials['cells'].data

    @cell_materials.setter
    def cell_materials(self, materials: Sequence[int]):
        """Material indexes for each cell."""
        self._cell_materials = make_material_dataset(materials)

    @property
    def grid_hash(self) -> str:
        """
        The hash of the UGrid that was linked to the simulation when materials were mapped.

        If uninitialized, defaults to an empty string.
        """
        return self.info.attrs.get('grid_hash', '')

    @grid_hash.setter
    def grid_hash(self, value: str):
        """
        The hash of the UGrid that was linked to the simulation when materials were mapped.

        If uninitialized, defaults to an empty string.
        """
        self.info.attrs['grid_hash'] = value

    def commit(self):
        """Save in memory datasets to the NetCDF file."""
        super().commit()
        self._cell_materials.to_netcdf(self._filename, group='cell_materials', mode='a')
