"""Class to parse a .2dm file."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.coverage.xy.xy_series import XySeries
from xms.gmi.data.generic_model import Type

# 4. Local modules
from xms.hydroas.file_io.errors import error, Messages as Msg
from xms.hydroas.file_io.gmi_fixer import ENTITY_NAMES
from xms.hydroas.file_io.gmi_parser import (
    BC_VAL_CARD_FIRST_VARIABLE_FIELD, BEFONT_CARD_SHORT_FONT_INDEX, BEFONT_CARD_SHORT_LENGTH, CARD_INDEX,
    DEF_CARD_FIRST_VARIABLE_FIELD, DEF_CARD_FIXED_FIELDS, DEP_CARD_FILLER_INDEX, DEP_CARD_FIRST_OPTION_INDEX,
    FIRST_FIELD, GP_VAL_CARD_FIRST_VARIABLE_FIELD, MAT_VAL_CARD_FIRST_VARIABLE_FIELD, OPTS_CARD_FIRST_OPTION_INDEX,
    parse_field, SI_CARD_MAX_LENGTH
)


def validate(lines: list, curves: dict[int, XySeries]) -> None:
    """
    Validate a parsed GMI definition.

    Raises GmiError if any errors are present.

    Args:
        lines: Parsed lines from the file.
        curves: Available curves.
    """
    validator = GmiValidator(lines, curves)
    validator.validate()


def _card_index_to_field_number(index: int):
    """
    Convert an index in a line into the field to be displayed to the user.

    Args:
        index: Index to convert.

    Returns:
        The converted index.
    """
    return index


class GmiValidator:
    """Validates the GMI part of a .2dm file."""
    def __init__(self, cards: list, curves: dict[int, XySeries]):
        """Initialize the validator.

        Params:
            cards: List of parsed lines in the file.
        """
        self._curves: dict[int, XySeries] = curves
        self._cards = cards
        self._card = []
        self._card_number = 0
        self._validators = {
            'bc': self._validate_bc,
            'bc_def': self._validate_def,
            'bc_dep': self._validate_dep,
            'bc_opts': self._validate_opts,
            'bc_disp_opts': self._validate_bc_disp_opts,
            'bc_val': self._validate_bc_val,
            'bcpgc': self._null_validator,
            'bedisp': self._validate_bedisp,
            'befont': self._validate_befont,
            'beg2dmbc': self._null_validator,
            'begcurve': self._null_validator,
            'begparamdef': self._null_validator,
            'e3t': self._validate_e3t,
            'e4q': self._validate_e4q,
            'end2dmbc': self._null_validator,
            'endcurve': self._null_validator,
            'endparamdef': self._null_validator,
            'disp_opts': self._validate_disp_opts,
            'dy': self._check_duplicate,
            'gm': self._check_duplicate,
            'gp': self._validate_gp,
            'gp_def': self._validate_def,
            'gp_dep': self._validate_dep,
            'gp_opts': self._validate_opts,
            'gp_val': self._validate_gp_val,
            'key': self._check_duplicate,
            'mat': self._validate_mat,
            'mat_def': self._validate_def,
            'mat_dep': self._validate_dep,
            'mat_multi': self._validate_mat_multi,
            'mat_opts': self._validate_opts,
            'mat_params': self._validate_mat_params,
            'mat_val': self._validate_mat_val,
            'mesh2d': self._null_validator,
            'meshname': self._null_validator,
            'nd': self._validate_nd,
            'num_materials_per_elem': self._validate_num_materials_per_elem,
            'nume': self._null_validator,
            'ns': self._validate_ns,
            'si': self._validate_si,
            'td': self._validate_td,
            'tu': self._check_duplicate,
        }
        self._seen_card_names = {}

        self._defined_nodes = {}
        self._referenced_nodes = {}
        self._defined_nodestrings = {}
        self._defined_elements = {}
        self._referenced_materials = {}
        self._dep_cards = []  # (index, entity_id)

        self._gp_group_counts = {}  # group_name -> number_seen
        self._seen_gp_groups = {}  # (name or ID) -> line number
        self._seen_gp_params = {}  # (group_id, (param_id or name)) -> line number
        self._gp_param_types = {}  # (group_id, param_id) -> Type
        self._gp_options = {}  # (group_id, param_id) -> list of valid options for this type
        self._gp_defaults = {}  # (group_id, param_id) -> default value for option type
        self._gp_name_id_map = {}  # (group_id, param_id) | (group_id, param_name) -> param_name | param_id
        self._gp_set_params = {}  # (group_id, param_id) -> line number
        self._gp_limits = {}  # (group_id, param_id) -> (low, high)

        self._last_entity_id = None
        self._seen_bc_groups = {}
        self._seen_bc_params = {}
        self._bc_param_types = {}
        self._bc_options = {}
        self._bc_defaults = {}
        self._bc_name_id_map = {}
        self._bc_set_params = {}
        self._bc_limits = {}

        self._seen_mat_groups = {}  # 0 is the disable ID. It's always implicitly defined.
        self._seen_mat_params = {}
        self._mat_param_types = {}
        self._mat_options = {}
        self._mat_defaults = {}
        self._mat_name_id_map = {}
        self._mat_set_params = {}
        self._mat_limits = {}

    def validate(self):
        """Run the validator."""
        for card_number, card in enumerate(self._cards, start=1):
            if not card:
                continue
            self._card_number = card_number
            card_name = card[CARD_INDEX]
            self._card = card
            validator = self._validators[card_name]
            validator()

        self._check_references()
        self._check_dep_cards()

    def _check_duplicate(self):
        """Check if a card has been used more than once."""
        card_name = self._card[CARD_INDEX]
        self._check_standard_name(self._card_number, card_name)

    def _check_references(self):
        """Check that nodes and materials referenced by E3T and E4Q cards exist."""
        # Material 0 can't be explicitly defined, so any use of it will be flagged as using an undefined ID.
        # But material 0 is implicitly defined, so all references to it are automatically valid.
        # We'll just pretend nobody references it, to go along with nobody defining it.
        self._referenced_materials.pop(0, None)

        checks = [(self._defined_nodes, self._referenced_nodes), (self._seen_mat_groups, self._referenced_materials)]
        report_line = report_field = None

        for definitions, references in checks:
            defined_ids = definitions.keys()
            referenced_ids = references.keys()
            missing_ids = referenced_ids - defined_ids

            for missing_id in missing_ids:
                line_number, field_number = references[missing_id]
                # If there are multiple errors, report the one at the lowest line number.
                if report_line is None or line_number < report_line:
                    report_line = line_number
                    report_field = field_number

        if report_line is not None:
            error(report_line, report_field, Msg.undefined_id)

    def _check_standard_name(self, line_number: int, card_name: str):
        """
        Check that a card has only been used once.

        Args:
            line_number: The line the card to check was used on.
            card_name: The name of the card to check, e.g. 'TY' or 'KEY'.
        """
        if card_name in self._seen_card_names:
            original_line = self._seen_card_names[card_name]
            error(line_number, Msg.duplicate_card, original_line)
        self._seen_card_names[card_name] = line_number

    def _null_validator(self):
        """Validate a card that can't possibly be invalid."""
        pass

    def _validate_bc(self):
        """Validate a BC card."""
        _card_name, entity_id, bc_name, bc_id, filler, _, gp_group_name = self._card
        if entity_id not in ENTITY_NAMES:
            error(self._card_number, 1, Msg.range_error)
        if filler != 0:
            error(self._card_number, 4, Msg.range_error)
        if bc_id < 1:
            error(self._card_number, 3, Msg.range_error)

        bc_id = (entity_id, bc_id)
        bc_name = (entity_id, bc_name)

        if bc_id in self._seen_bc_groups:
            original_line = self._seen_bc_groups[bc_id]
            error(self._card_number, 3, Msg.duplicate_id, original_line)
        if gp_group_name != "(none)" and gp_group_name != '' and gp_group_name not in self._seen_gp_groups:
            error(self._card_number, 6, Msg.undefined_group_name)
        if self._gp_group_counts.get(gp_group_name, 0) > 1:
            error(self._card_number, 6, Msg.ambiguous_correlation)

        self._last_entity_id = entity_id
        self._seen_bc_groups[bc_id] = self._card_number
        self._seen_bc_groups[bc_name] = self._card_number

    def _validate_bc_val(self):
        """Validate a BC_VAL card."""
        card_name, entity_id, item_id, bc_id, param_id = self._card[:BC_VAL_CARD_FIRST_VARIABLE_FIELD]

        if entity_id == 'points':
            definitions = self._defined_nodes
        elif entity_id == 'arcs':
            definitions = self._defined_nodestrings
        elif entity_id == 'polygons':
            definitions = self._defined_elements
        else:
            error(self._card_number, 1, Msg.bad_entity_short_name)

        bc_id = (entity_id, bc_id)

        if bc_id not in self._seen_bc_groups:
            error(self._card_number, 1, Msg.undefined_id)
        if (bc_id, param_id) not in self._seen_bc_params:
            error(self._card_number, 2, Msg.undefined_id)
        if (entity_id, item_id, bc_id, param_id) in self._bc_set_params:
            orig_line = self._bc_set_params[(entity_id, item_id, bc_id, param_id)]
            error(self._card_number, Msg.duplicate_card, orig_line)
        if item_id not in definitions:
            error(self._card_number, 2, Msg.undefined_id)

        self._bc_set_params[(entity_id, item_id, bc_id, param_id)] = self._card_number
        param_type = self._bc_param_types[(bc_id, param_id)]

        limits = self._bc_limits[(bc_id, param_id)] if (bc_id, param_id) in self._bc_limits else []
        options = self._bc_options[(bc_id, param_id)] if (bc_id, param_id) in self._bc_options else []
        self._validate_val(param_type, BC_VAL_CARD_FIRST_VARIABLE_FIELD, limits, options)

    def _validate_befont(self):
        """Validate a BEFONT card."""
        card_name = self._card[CARD_INDEX]
        entity_id = self._card[FIRST_FIELD]
        card_name = f'{card_name},{entity_id}'
        self._check_standard_name(self._card_number, card_name)

        if entity_id not in ENTITY_NAMES:
            error(self._card_number, _card_index_to_field_number(1), Msg.range_error)

        if len(self._card) == BEFONT_CARD_SHORT_LENGTH and not (1 <= self._card[BEFONT_CARD_SHORT_FONT_INDEX] <= 2):
            error(self._card_number, _card_index_to_field_number(BEFONT_CARD_SHORT_FONT_INDEX), Msg.range_error)

    def _validate_def(self):
        """Validate a *_DEF card."""
        card_name, group_id, param_id, param_name, param_type = self._card[:DEF_CARD_FIXED_FIELDS]

        if card_name == 'gp_def':
            seen_groups = self._seen_gp_groups
            seen_params = self._seen_gp_params
            param_types = self._gp_param_types
            name_id_map = self._gp_name_id_map
            limits = self._gp_limits
            defaults = self._gp_defaults
        elif card_name == 'bc_def':
            seen_groups = self._seen_bc_groups
            seen_params = self._seen_bc_params
            param_types = self._bc_param_types
            name_id_map = self._bc_name_id_map
            limits = self._bc_limits
            defaults = self._bc_defaults
        elif card_name == 'mat_def':
            seen_groups = self._seen_gp_groups  # Materials associate with global groups. They don't have their own.
            seen_params = self._seen_mat_params
            param_types = self._mat_param_types
            name_id_map = self._mat_name_id_map
            limits = self._mat_limits
            defaults = self._mat_defaults
        else:
            # This is programmer error.
            raise AssertionError("Unrecognized card")  # pragma: nocover

        if group_id < 1:
            error(self._card_number, 1, Msg.range_error)

        if card_name == 'bc_def':
            group_id = (self._last_entity_id, group_id)

        if group_id not in seen_groups and card_name != 'mat_def':
            error(self._card_number, 1, Msg.undefined_id)
        elif group_id not in seen_groups:
            seen_groups[group_id] = self._card_number  # MAT_DEF groups are implicitly defined.
        if param_id < 0:
            error(self._card_number, 2, Msg.range_error)
        if (group_id, param_id) in seen_params:
            original_line = seen_params[(group_id, param_id)]
            error(self._card_number, Msg.duplicate_id, original_line)
        if not (0 <= param_type <= 6):
            # The parser should have rejected this, or a new type was added without support here.
            raise AssertionError("Unknown type")  # pragma: nocover

        name_id_map[(group_id, param_id)] = param_name
        name_id_map[(group_id, param_name)] = param_id

        seen_params[(group_id, param_id)] = self._card_number
        seen_params[(group_id, param_name)] = self._card_number

        if param_type == 0:
            self._validate_def_bool(param_types)
        elif param_type == 1:
            self._validate_def_int(param_types, limits)
        elif param_type == 2:
            self._validate_def_float(param_types, limits)
        elif param_type == 3:
            self._validate_def_text(param_types)
        elif param_type == 4:
            self._validate_def_option(param_types, defaults)
        elif param_type == 5:
            self._validate_def_curve(param_types)
        elif param_type == 6:
            self._validate_def_float_curve(param_types, limits)
        else:
            # This type needs to be implemented to use it.
            raise AssertionError("Unrecognized type")  # pragma: nocover

    def _validate_def_bool(self, param_types):
        """
        Validate a *_DEF card for a boolean parameter.

        Args:
            param_types: Where to store the parameter's type.
        """
        card_name, group_id, param_id, param_name, _type, _default_value = self._card
        if card_name == 'bc_def':
            group_id = (self._last_entity_id, group_id)
        param_types[(group_id, param_id)] = Type.BOOLEAN

    def _validate_def_curve(self, param_types):
        """
        Validate a *_DEF card for a curve parameter.

        Args:
            param_types: Where to store the parameter's type.
        """
        card_name, group_id, param_id, param_name, _type = self._card[:DEF_CARD_FIRST_VARIABLE_FIELD]
        if card_name == 'bc_def':
            group_id = (self._last_entity_id, group_id)
        param_types[(group_id, param_id)] = Type.CURVE

    def _validate_def_float(self, param_types, limits):
        """
        Validate a *_DEF card for a float parameter.

        Args:
            param_types: Where to store the parameter's type.
            limits: Where to store the parameter's limits.
        """
        card_name, group_id, param_id, param_name, _type, default_value, low, high = self._card
        if card_name == 'bc_def':
            group_id = (self._last_entity_id, group_id)
        if low > high:
            error(self._card_number, Msg.range_inverted)
        if not (low <= default_value <= high):
            error(self._card_number, 5, Msg.range_error_defined)
        param_types[(group_id, param_id)] = Type.FLOAT
        limits[(group_id, param_id)] = (low, high)

    def _validate_def_float_curve(self, param_types, limits):
        """
        Validate a *_DEF card for a flaot-curve parameter.

        Args:
            param_types: Where to store the parameter's type.
            limits: Where to store the parameter's limits.
        """
        card_name, group_id, param_id, param_name, _type, default_value, low, high, mode, _x, _y = self._card
        if card_name == 'bc_def':
            group_id = (self._last_entity_id, group_id)
        if low > high:
            error(self._card_number, Msg.range_inverted)
        if not (low <= default_value <= high):
            error(self._card_number, 5, Msg.range_error_defined)
        if mode not in ['FLOAT', 'CURVE', 'VALUE']:
            error(self._card_number, 8, Msg.not_float_or_curve)
        param_types[(group_id, param_id)] = Type.FLOAT_CURVE
        limits[(group_id, param_id)] = (low, high)

    def _validate_def_int(self, param_types, limits):
        """
        Validate a *_DEF card for an integer parameter.

        Args:
            param_types: Where to store the parameter's type.
            limits: Where to store the parameter's limits.
        """
        card_name, group_id, param_id, param_name, _type, default_value, low, high = self._card
        if card_name == 'bc_def':
            group_id = (self._last_entity_id, group_id)
        if low > high:
            error(self._card_number, Msg.range_inverted)
        if not (low <= default_value <= high):
            error(self._card_number, 5, Msg.range_error_defined)
        param_types[(group_id, param_id)] = Type.INTEGER
        limits[(group_id, param_id)] = (low, high)

    def _validate_def_option(self, param_types, defaults):
        """
        Validate a *_DEF card for an option parameter.

        Args:
            param_types: Where to store the parameter's type.
            defaults: Where to store the parameter's default value.
        """
        card_name, group_id, param_id, param_name, _type = self._card[:DEF_CARD_FIRST_VARIABLE_FIELD]
        if card_name == 'bc_def':
            group_id = (self._last_entity_id, group_id)
        param_types[(group_id, param_id)] = Type.OPTION
        defaults[(group_id, param_id)] = self._card[DEF_CARD_FIRST_VARIABLE_FIELD]

    def _validate_def_text(self, param_types):
        """
        Validate a *_DEF card for a text parameter.

        Args:
            param_types: Where to store the parameter's type.
        """
        card_name, group_id, param_id, param_name, _type = self._card[:DEF_CARD_FIRST_VARIABLE_FIELD]
        if card_name == 'bc_def':
            group_id = (self._last_entity_id, group_id)
        param_types[(group_id, param_id)] = Type.TEXT

    def _validate_dep(self):
        """Validate a *_DEP card."""
        # It's possible for a *_DEP card to appear before the card defining its parent. Since we might not know about
        # the parent yet, we'll store off the *_DEP card and check it after we've found all the *_DEF ones.
        card_name = self._card[0]
        if card_name == 'gp_dep':
            entity_id = 'global'
        elif card_name == 'mat_dep':
            entity_id = 'material'
        elif card_name == 'bc_dep':
            entity_id = self._last_entity_id
        else:
            raise AssertionError('Unknown card')  # pragma: nocover
        self._dep_cards.append((self._card_number - 1, entity_id))

    def _check_dep_cards(self):
        """
        Check that all the *_DEP cards are valid.

        This method assumes all the relevant *_DEF cards have already been validated, so it should only be called after.
        """
        for index, entity_id in self._dep_cards:
            self._card_number = index + 1
            self._card = self._cards[index]
            self._last_entity_id = entity_id
            self._check_dep_card()

    def _check_dep_card(self):
        """
        Check that a *_DEP card is valid.

        This method assumes all the relevant *_DEF cards have already been validated, so it should only be called after.
        """
        card_name, group_id, param_id, dependency_type, parent_name = self._card[:DEP_CARD_FILLER_INDEX]

        if card_name == 'bc_dep':
            group_id = (self._last_entity_id, group_id)
            options = self._bc_options
            param_types = self._bc_param_types
            seen_groups = self._seen_bc_groups
            seen_params = self._seen_bc_params
            name_id_map = self._bc_name_id_map
        elif card_name == 'gp_dep':
            options = self._gp_options
            param_types = self._gp_param_types
            seen_groups = self._seen_gp_groups
            seen_params = self._seen_gp_params
            name_id_map = self._gp_name_id_map
        elif card_name == 'mat_dep':
            options = self._mat_options
            param_types = self._mat_param_types
            seen_groups = self._seen_gp_groups  # Materials associate with global groups. They don't have their own.
            seen_params = self._seen_mat_params
            name_id_map = self._mat_name_id_map
        else:
            # Called for a card this doesn't support.
            raise ValueError("Unrecognized card")  # pragma: nocover

        if group_id not in seen_groups:
            error(self._card_number, 1, Msg.undefined_id)
        if (group_id, param_id) not in seen_params:
            error(self._card_number, 2, Msg.undefined_id)
        if dependency_type not in ['PARENT_GLOBAL', 'PARENT_NONE', 'PARENT_LOCAL', 'PARENT_SELF']:
            error(self._card_number, 3, Msg.bad_dependency_type)
        if dependency_type == 'PARENT_GLOBAL':
            error(self._card_number, 3, Msg.bad_dependency_type)

        found_parents = 0
        found_parent = None

        if dependency_type == 'PARENT_NONE':
            return  # No parent, so no need to validate.

        # If someone complains about PARENT_GLOBAL, this is how it should work. It's poorly designed though, so we'd
        # rather just not support it if we can get away with it.
        # elif dependency_type == 'PARENT_GLOBAL':
        #     for group_id, name in seen_params:
        #         if name == parent_name:
        #             found_parents += 1
        #             param_id = name_id_map[(group_id, name)]
        #             found_parent = (group_id, param_id)

        elif dependency_type == 'PARENT_LOCAL' or dependency_type == 'PARENT_SELF':
            for seen_group, seen_id in seen_params:
                if seen_group != group_id:
                    continue
                if name_id_map[(seen_group, seen_id)] != parent_name:
                    continue
                if param_types[(seen_group, seen_id)] != Type.OPTION:
                    continue
                found_parents += 1
                found_parent = (seen_group, seen_id)

        else:
            # Unsupported type was allowed through.
            raise AssertionError("Unrecognized dependency type.")  # pragma: nocover

        if found_parents == 0:
            error(self._card_number, 4, Msg.missing_parent)
        if found_parents > 1:
            error(self._card_number, Msg.ambiguous_parent)

        available_options = set(options[found_parent])
        for i in range(DEP_CARD_FIRST_OPTION_INDEX, len(self._card), 2):
            if self._card[i] not in available_options:
                display_offset = _card_index_to_field_number(i)
                error(self._card_number, display_offset, Msg.missing_parent)

    def _validate_disp_opts(self):
        """Validate a DISP_OPTS card."""
        card_name, section, entity_id, red, green, blue, display, pattern, width, style = self._card
        card_name = f'{card_name},{section},{entity_id}'
        self._check_standard_name(self._card_number, card_name)

        if section not in ['entity', 'inactive', 'multiple']:
            error(self._card_number, 1, Msg.bad_entity_group)
        if entity_id not in ENTITY_NAMES:
            error(self._card_number, 2, Msg.range_error)
        if not (0 <= red <= 255):
            error(self._card_number, 3, Msg.range_error)
        if not (0 <= green <= 255):
            error(self._card_number, 4, Msg.range_error)
        if not (0 <= blue <= 255):
            error(self._card_number, 5, Msg.range_error)

    def _validate_e3t(self):
        """Validate an E3T card."""
        _card_name, element_id, first, second, third, material_id = self._card

        if element_id < 1:
            error(self._card_number, 1, Msg.range_error)

        if element_id in self._defined_elements:
            original_line = self._defined_elements[element_id]
            error(self._card_number, 1, Msg.duplicate_id, original_line)

        self._defined_elements[element_id] = self._card_number

        # Use setdefault so we only keep the line that first referenced the node.
        self._referenced_nodes.setdefault(first, (self._card_number, 2))
        self._referenced_nodes.setdefault(second, (self._card_number, 3))
        self._referenced_nodes.setdefault(third, (self._card_number, 4))
        self._referenced_materials.setdefault(material_id, (self._card_number, 5))

    def _validate_e4q(self):
        """Validate an E4Q card."""
        _card_name, element_id, first, second, third, fourth, material_id = self._card

        if element_id < 1:
            error(self._card_number, 1, Msg.range_error)

        if element_id in self._defined_elements:
            original_line = self._defined_elements[element_id]
            error(self._card_number, 1, Msg.duplicate_id, original_line)

        self._defined_elements[element_id] = self._card_number

        # Use setdefault so we only keep the line that first referenced the node.
        self._referenced_nodes.setdefault(first, (self._card_number, 2))
        self._referenced_nodes.setdefault(second, (self._card_number, 3))
        self._referenced_nodes.setdefault(third, (self._card_number, 4))
        self._referenced_nodes.setdefault(fourth, (self._card_number, 5))
        self._referenced_materials.setdefault(material_id, (self._card_number, 6))

    def _validate_bc_disp_opts(self):
        """Validate a BC_DISP_OPTS card."""
        _card_name, bc_id, entity_id, red, green, blue, _display, pattern, width, style = self._card

        if entity_id not in ENTITY_NAMES:
            error(self._card_number, 1, Msg.range_error)
        if (entity_id, bc_id) not in self._seen_bc_groups:
            error(self._card_number, 1, Msg.undefined_id)
        if not (0 <= red <= 255):
            error(self._card_number, 2, Msg.range_error)
        if not (0 <= green <= 255):
            error(self._card_number, 3, Msg.range_error)
        if not (0 <= blue <= 255):
            error(self._card_number, 4, Msg.range_error)

    def _validate_bedisp(self):
        """Validate a BEDISP card."""
        _card_name, entity_id, font_red, font_green, font_blue = self._card[:5]
        label_on, label_vals_on, inactive_size, inactive_style = self._card[5:9]
        inactive_red, inactive_green, inactive_blue, inactive_on = self._card[9:]

        if not (0 <= entity_id <= 2):
            error(self._card_number, 1, Msg.range_error)
        if not (0 <= font_red <= 255):
            error(self._card_number, 2, Msg.range_error)
        if not (0 <= font_green <= 255):
            error(self._card_number, 3, Msg.range_error)
        if not (0 <= font_blue <= 255):
            error(self._card_number, 4, Msg.range_error)
        if entity_id == 0 and not (1 <= inactive_style <= 11):
            error(self._card_number, 8, Msg.range_error)
        if entity_id != 0 and not (0 <= inactive_style <= 1):
            error(self._card_number, 8, Msg.range_error)
        if not (0 <= inactive_red <= 255):
            error(self._card_number, 9, Msg.range_error)
        if not (0 <= inactive_green <= 255):
            error(self._card_number, 10, Msg.range_error)
        if not (0 <= inactive_blue <= 255):
            error(self._card_number, 11, Msg.range_error)

    def _validate_gp(self):
        """Validate a GP card."""
        _card_name, group_id, group_name, _ = self._card
        if group_id in self._seen_gp_groups:
            original_line = self._seen_gp_groups[group_id]
            error(self._card_number, 1, Msg.duplicate_id, original_line)
        self._seen_gp_groups[group_id] = self._card_number
        self._seen_gp_groups[group_name] = self._card_number
        self._gp_group_counts[group_name] = self._gp_group_counts.get(group_name, 0) + 1

        if group_id < 1:
            error(self._card_number, 2, Msg.range_error)

    def _validate_gp_val(self):
        """Validate a GP_VAL card."""
        card_name, group_id, param_id = self._card[:GP_VAL_CARD_FIRST_VARIABLE_FIELD]

        if group_id not in self._seen_gp_groups:
            error(self._card_number, 1, Msg.undefined_id)
        if (group_id, param_id) not in self._seen_gp_params:
            error(self._card_number, 2, Msg.undefined_id)
        if (group_id, param_id) in self._gp_set_params:
            orig_line = self._gp_set_params[(group_id, param_id)]
            error(self._card_number, Msg.duplicate_card, orig_line)

        self._gp_set_params[(group_id, param_id)] = self._card_number
        param_type = self._gp_param_types[(group_id, param_id)]

        limits = self._gp_limits[(group_id, param_id)] if (group_id, param_id) in self._gp_limits else []
        options = self._gp_options[(group_id, param_id)] if (group_id, param_id) in self._gp_options else []
        self._validate_val(param_type, GP_VAL_CARD_FIRST_VARIABLE_FIELD, limits, options)

    def _validate_mat(self):
        """Validate a MAT card."""
        _card_name, mat_id, mat_name = self._card

        if mat_id == 0:
            error(self._card_number, 1, Msg.defined_disabled_material)
        if mat_id in self._seen_mat_groups:
            original_line = self._seen_mat_groups[mat_id]
            error(self._card_number, 1, Msg.duplicate_id, original_line)

        self._seen_mat_groups[mat_id] = self._card_number
        self._seen_mat_groups[mat_name] = self._card_number

    def _validate_mat_multi(self):
        """Validate a MAT_MULTI card."""
        _card_name, is_set, group_id = self._card
        if is_set and not group_id:
            error(self._card_number, Msg.missing_field)
        if is_set and group_id not in self._seen_gp_groups:
            error(self._card_number, 2, Msg.undefined_id)

    def _validate_mat_params(self):
        """Validate a MAT_PARAMS card."""
        _card_name, material_id, group_id = self._card
        if material_id not in self._seen_mat_groups:
            error(self._card_number, 1, Msg.undefined_id)
        if group_id not in self._seen_gp_groups:
            error(self._card_number, 2, Msg.undefined_id)

    def _validate_mat_val(self):
        """Validate a MAT_VAL card."""
        card_name, mat_id, group_id, param_id = self._card[:MAT_VAL_CARD_FIRST_VARIABLE_FIELD]

        if mat_id not in self._seen_mat_groups:
            error(self._card_number, 1, Msg.undefined_id)
        if group_id not in self._seen_gp_groups:
            error(self._card_number, 2, Msg.undefined_id)
        if (group_id, param_id) not in self._seen_mat_params:
            error(self._card_number, 3, Msg.undefined_id)
        if (mat_id, group_id, param_id) in self._mat_set_params:
            orig_line = self._mat_set_params[(mat_id, group_id, param_id)]
            error(self._card_number, Msg.duplicate_card, orig_line)

        self._mat_set_params[(mat_id, group_id, param_id)] = self._card_number
        param_type = self._mat_param_types[(group_id, param_id)]

        limits = self._mat_limits[(group_id, param_id)] if (group_id, param_id) in self._mat_limits else []
        options = self._mat_options[(group_id, param_id)] if (group_id, param_id) in self._mat_options else []
        self._validate_val(param_type, MAT_VAL_CARD_FIRST_VARIABLE_FIELD, limits, options)

    def _validate_nd(self):
        """Validate an ND card."""
        node_id = self._card[FIRST_FIELD]

        if node_id < 1:
            error(self._card_number, 1, Msg.range_error)

        if node_id in self._defined_nodes:
            original_line = self._defined_nodes[node_id]
            error(self._card_number, 1, Msg.duplicate_id, original_line)

        self._defined_nodes[node_id] = self._card_number

    def _validate_ns(self):
        """Validate an NS card."""
        nodes = self._card[FIRST_FIELD:]
        nodes.pop()  # The last element is always a name. The fixer enforces this by appending blank ones if necessary.

        # Our string should look something like [int, int, int, int, -int, int] now since the fixer concatenates
        # multi-line strings. Check if it doesn't.
        negatives = [index for index, node in enumerate(nodes) if node < 0]
        if len(negatives) == 0:
            error(self._card_number, 0, Msg.missing_negative_node, self._card_number)
        elif len(negatives) > 1:
            error(self._card_number, FIRST_FIELD + negatives[1], Msg.extra_negative_node)

        negative = negatives[0]
        second_to_last = len(nodes) - 2
        if negative < second_to_last:
            # There should be one positive after the negative, so we want to complain about the one after.
            error(self._card_number, FIRST_FIELD + negative + 2, Msg.node_after_end)
        elif negative > second_to_last:
            error(self._card_number, Msg.missing_nodestring_id)

        nodestring_id_index = len(nodes) - 1
        nodestring_id = nodes.pop()
        if nodestring_id == 0:
            error(self._card_number, FIRST_FIELD + nodestring_id_index, Msg.range_error)
        elif nodestring_id in self._defined_nodestrings:
            error(
                self._card_number, FIRST_FIELD + nodestring_id_index, Msg.duplicate_id,
                self._defined_nodestrings[nodestring_id]
            )
        else:
            self._defined_nodestrings[nodestring_id] = self._card_number

        if len(nodes) < 2:
            error(self._card_number, Msg.short_nodestring)

        nodes[-1] = -nodes[-1]
        for index, node in enumerate(nodes, start=1):
            self._referenced_nodes.setdefault(node, (self._card_number, index))

    def _validate_num_materials_per_elem(self):
        """Validate a NUM_MATERIALS_PER_ELEM card."""
        materials_per_elem = self._card[FIRST_FIELD]
        if materials_per_elem != 1:
            error(self._card_number, 1, Msg.range_error)

    def _validate_opts(self):
        """Validate a *_OPTS card."""
        card_name, group_id, param_id = self._card[:OPTS_CARD_FIRST_OPTION_INDEX]
        if card_name == 'bc_opts':
            group_id = (self._last_entity_id, group_id)
            seen_params = self._seen_bc_params
            param_types = self._bc_param_types
            param_options = self._bc_options
            defaults = self._bc_defaults
        elif card_name == 'gp_opts':
            seen_params = self._seen_gp_params
            param_types = self._gp_param_types
            param_options = self._gp_options
            defaults = self._gp_defaults
        elif card_name == 'mat_opts':
            seen_params = self._seen_mat_params
            param_types = self._mat_param_types
            param_options = self._mat_options
            defaults = self._mat_defaults
        else:
            # Unimplemented card got through.
            raise AssertionError("Unrecognized card")  # pragma: nocover

        if (group_id, param_id) not in seen_params:
            error(self._card_number, Msg.undefined_target)
        target_type = param_types[(group_id, param_id)]
        if target_type != Type.OPTION:
            error(self._card_number, Msg.target_not_option)

        options = self._card[OPTS_CARD_FIRST_OPTION_INDEX:]

        default = defaults[(group_id, param_id)]
        if default not in options:
            error(self._card_number, Msg.option_bad_default)

        param_options[(group_id, param_id)] = options

    def _validate_si(self):
        """Validate an SI card."""
        self._check_duplicate()
        if len(self._card) == SI_CARD_MAX_LENGTH:
            _card, units, flag = self._card
        else:
            _card, units = self._card
            flag = ''
        if not (0 <= units <= 2):
            error(self._card_number, 2, Msg.range_error)
        if flag not in ['', 'International']:
            error(self._card_number, 3, Msg.not_international)
        if units != 0 and flag:
            error(self._card_number, 3, Msg.bad_international)

    def _validate_td(self):
        """Validate a TD card."""
        self._check_duplicate()
        _card_name, time_step, total_time = self._card
        if time_step < 0:
            error(self._card_number, 1, Msg.range_error)
        if total_time < 0:
            error(self._card_number, 2, Msg.range_error)

    def _validate_val(self, param_type: Type, field_offset: int, limits: tuple[int, int], options: list[str]):
        """
        Validate a *_VAL card.

        Args:
            param_type: Type of parameter the *_VAL card is assigning to.
            field_offset: Index into the card where the first variable field is.
            limits: Min and max value. Only used for numeric types.
            options: Available options. Only used for option types.
        """
        # The parser knows *_VAL cards need either 1 or 2 fields for the value, and rejects everything else.
        # So at this point, we have either 1 or 2 fields.
        # The parser doesn't have type information though, so it can't tell which of 1 or 2 fields it needs.
        # We need to deal with that here, where type information is available.
        if param_type == Type.FLOAT_CURVE and len(self._card) < field_offset + 2:
            error(self._card_number, Msg.missing_field)
        elif param_type != Type.FLOAT_CURVE and len(self._card) > field_offset + 1:
            error(self._card_number, Msg.extra_field)

        if param_type == Type.BOOLEAN:
            self._validate_val_bool(field_offset)
        elif param_type == Type.INTEGER:
            self._validate_val_int(field_offset, limits)
        elif param_type == Type.FLOAT:
            self._validate_val_float(field_offset, limits)
        elif param_type == Type.TEXT:
            self._validate_val_text(field_offset)
        elif param_type == Type.OPTION:
            self._validate_val_option(field_offset, options)
        elif param_type == Type.CURVE:
            self._validate_val_curve(field_offset)
        elif param_type == Type.FLOAT_CURVE:
            self._validate_val_float_curve(field_offset, limits)
        else:
            # New type added, but not supported here.
            raise AssertionError("Unrecognized type")  # pragma: nocover

    def _validate_val_bool(self, field_offset: int):
        """
        Validate a *_VAL card for a boolean parameter.

        Args:
            field_offset: Index into the card where the first field is.
        """
        self._validate_val_fields(field_offset, 'b')

    def _validate_val_curve(self, field_offset: int):
        """
        Validate a *_VAL card for a curve parameter.

        Args:
            field_offset: Index into the card where the first field is.
        """
        self._validate_val_fields(field_offset, 'i')
        curve_id = self._card[field_offset]
        if curve_id not in self._curves and curve_id != -1:
            display_offset = _card_index_to_field_number(field_offset)
            error(self._card_number, display_offset, Msg.undefined_id)

    def _validate_val_fields(self, field_offset: int, pattern: str):
        """
        Parse the fields of a *_VAL card.

        Args:
            field_offset: Index into the card where the first field is.
            pattern: Pattern to parse with. See parse_field().
        """
        display_offset = _card_index_to_field_number(field_offset)
        self._card[field_offset] = parse_field(self._card_number, self._card[field_offset], pattern, display_offset)

    def _validate_val_float(self, field_offset: int, limits: tuple[int, int]):
        """
        Validate a *_VAL card for a float parameter.

        Args:
            field_offset: Index into the card where the first field is.
            limits: Min and max value.
        """
        self._validate_val_fields(field_offset, 'f')
        low, high = limits
        if not (low <= self._card[field_offset] <= high):
            display_offset = _card_index_to_field_number(field_offset)
            error(self._card_number, display_offset, Msg.range_error_defined)

    def _validate_val_int(self, field_offset: int, limits: tuple[int, int]):
        """
        Validate a *_VAL card for an int parameter.

        Args:
            field_offset: Index into the card where the first field is.
            limits: Min and max value.
        """
        self._validate_val_fields(field_offset, 'i')
        low, high = limits
        if not (low <= self._card[field_offset] <= high):
            display_offset = _card_index_to_field_number(field_offset)
            error(self._card_number, display_offset, Msg.range_error_defined)

    def _validate_val_float_curve(self, field_offset: int, limits: tuple[int, int]):
        """
        Validate a *_VAL card for a float-curve parameter.

        Args:
            field_offset: Index into the card where the first field is.
            limits: Min and max value for float mode.
        """
        # 'q' fields can only fail if the field has an opening quote but no closing one.
        # Our line came from the parser though, which would have rejected the whole file
        # if any field was like that, so this can't fail.
        self._card[field_offset] = parse_field(self._card_number, self._card[field_offset], 'q', -1)

        mode = self._card[field_offset]
        field_offset += 1

        if mode == 'FLOAT' or mode == 'VALUE':
            display_offset = _card_index_to_field_number(field_offset)
            self._card[field_offset] = parse_field(self._card_number, self._card[field_offset], 'f', display_offset)
            value = self._card[field_offset]
            low, high = limits
            if not (low <= value <= high):
                error(self._card_number, 4, Msg.range_error_defined)
        elif mode == 'CURVE':
            display_offset = _card_index_to_field_number(field_offset)
            self._card[field_offset] = parse_field(self._card_number, self._card[field_offset], 'i', display_offset)

            curve_id = self._card[field_offset]
            if curve_id not in self._curves and curve_id != -1:
                display_offset = _card_index_to_field_number(field_offset)
                error(self._card_number, display_offset, Msg.undefined_id)
        else:
            field_offset -= 1  # Undo the optimistic increment from above
            display_offset = _card_index_to_field_number(field_offset)
            error(self._card_number, display_offset, Msg.bad_mode)

    def _validate_val_option(self, field_offset: int, options: list[str]):
        """
        Validate a *_VAL card for an option parameter.

        Args:
            field_offset: Index into the card where the first field is.
            options: Allowed values for the option.
        """
        self._validate_val_fields(field_offset, 't')
        if self._card[field_offset] not in options:
            display_offset = _card_index_to_field_number(field_offset)
            error(self._card_number, display_offset, Msg.bad_option_value)

    def _validate_val_text(self, field_offset: int):
        """
        Validate a *_VAL card for a text parameter.

        Args:
            field_offset: Index into the card where the first field is.
        """
        self._validate_val_fields(field_offset, 't')
