"""ModelChecker class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from subprocess import PIPE, Popen, STDOUT

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg
from xms.guipy.dialogs.process_feedback_thread import ProcessFeedbackThread

# 4. Local modules
from xms.mf6.misc import log_util
from xms.mf6.simulation_runner import sim_runner


def _get_mf6_exe_path(exe_key: sim_runner.ExeKey) -> str:
    """Returns the path to the MODFLOW executable.

    Returns:
        See description.
    """
    paths = sim_runner.get_gms_mf6_executable_paths()
    return paths[exe_key]


class ModelChecker:
    """Runs the model checker."""
    def __init__(self, mfsim_dir, modflow_exe_string):
        """Initializes the class."""
        super().__init__()
        self._mfsim_dir = mfsim_dir
        self._mf6_exe_path = _get_mf6_exe_path(modflow_exe_string)
        self.run_dialog = True
        self._log = log_util.get_logger()
        self._in_errors = False
        self._in_warnings = False
        self._normal_termination = False

    def run_with_feedback_dialog(self, win_cont):
        """Runs the model checker using the ProcessFeedbackDialog.

        Args:
            win_cont (QWidget): The window container.
        """
        feedback = True
        if not feedback:
            self.do_work()  # pragma no cover - feedback will always be on except during testing
        else:
            try:
                mfsim_nam = os.path.join(self._mfsim_dir, 'mfsim.nam')
                worker = ProcessFeedbackThread(do_work=self.do_work, parent=win_cont)
                testing = win_cont is None
                auto_load_text = 'Close this dialog automatically when importing is finished.' if testing else ''
                display_text = {
                    'title': 'Check MODFLOW 6 Simulation',
                    'working_prompt': f'Checking the modflow simulation "{mfsim_nam}" for warnings or errors...',
                    'error_prompt': f'Error(s) encountered in the simulation {mfsim_nam}.',
                    'warning_prompt': f'Warning(s) encountered in the simulation {mfsim_nam}.',
                    'success_prompt': f'No warnings or errors encountered in the simulation {mfsim_nam}.',
                    'note': f'Using MODFLOW 6 executable: {self._mf6_exe_path}',
                    'auto_load': auto_load_text,
                    'log_format': '%(message)s',
                    'use_colors': True
                }
                feedback_dlg = ProcessFeedbackDlg(display_text=display_text, worker=worker, parent=win_cont)
                feedback_dlg.testing = testing
                feedback_dlg.exec()
            except Exception as error:  # pragma no cover - should never throw an exception
                raise RuntimeError(f'Error checking simulation: {str(error)}')  # pragma no cover

    def do_work(self) -> None:
        """Do the work."""
        try:
            os.chdir(self._mfsim_dir)
            with Popen(
                [f'{self._mf6_exe_path}', '-m', 'validate'], stdout=PIPE, stderr=STDOUT, universal_newlines=True
            ) as process:
                # mypy complains about this next line for some reason
                for line in process.stdout:  # type: ignore
                    self._handle_line(line.rstrip('\n'))

            if not self._normal_termination:
                self._log.error(' ')  # String must not be empty
        except Exception as ex:
            XmEnv.report_error(ex)

    def _handle_line(self, line):
        if line.find('is not recognized as an internal or external command') > -1:
            self._in_errors = True  # pragma no cover - if exe path is invalid should throw exception in do_work

        # See if we hit the beginning of the warnings or errors
        if not self._in_warnings and not self._in_errors and line.find('WARNING REPORT:') > -1:
            self._in_warnings = True
        elif not self._in_errors and (line.find('ERROR REPORT:') > -1 or line.find('UNIT ERROR REPORT:') > -1):
            self._in_errors = True
            self._in_warnings = False
        if line.find('Normal termination of simulation.') > -1:
            self._normal_termination = True

        # See if we hit the end of the warnings or errors
        if self._in_warnings and line.find('\n\n') > -1 or self._normal_termination:
            self._in_warnings = False
        elif self._in_errors and line.find('\n\n'):
            self._in_warnings = False

        # Log the line
        if self._in_warnings:
            self._log.warning(line)
        elif self._in_errors:
            self._log.error(line)
        else:
            self._log.info(line)

        return self._normal_termination
