"""CbcComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from collections import namedtuple
import os
from pathlib import Path
from typing import List, Optional

# 2. Third party modules
import orjson
from PySide2.QtWidgets import QDialog, QWidget
from typing_extensions import override

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util, TreeNode
from xms.guipy.dialogs import message_box, process_feedback_dlg
from xms.guipy.dialogs.feedback_thread import FeedbackThread

# 4. Local modules
from xms.mf6.components import cbc_velocity_vector_dataset_creator, dmi_util, flow_budget_runner, zone_budget_runner
from xms.mf6.components.cbc_scalar_dataset_creator import CbcScalarDatasetCreator
from xms.mf6.components.cbc_velocity_vector_dataset_creator import VectorInputs
from xms.mf6.components.mf6_component_base import Mf6ComponentBase
from xms.mf6.components.zone_budget_runner import ZbInput
from xms.mf6.data.grid_info import GridInfo
from xms.mf6.data.gwt.mst_data import MstData
from xms.mf6.file_io import grb_reader
from xms.mf6.file_io.grb_reader import GrbReader
from xms.mf6.gui.get_porosity_dialog import GetPorosityDialog
from xms.mf6.misc.settings import Settings
from xms.mf6.simulation_runner import sim_runner

# Type definitions
FlowBudgetDialogData = namedtuple('FlowBudgetDialogData', ['dataset_uuid', 'precision', 'data_file'])

# Constants
DEFAULT_PREC = 5  # Default precision for flow budget


class CbcComponent(Mf6ComponentBase):
    """A Dynamic Model Interface (DMI) component for a CBC MODFLOW 6 solution file."""

    prefix = 'zbud'  # Default used by ZONEBUDGET for the name file and output files

    def __init__(self, main_file):
        """Initializes the class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)

        self.build_vertex = None
        cards = {}
        with open(main_file.strip('\"'), 'rb') as file:
            cards = orjson.loads(file.read())
        self.sim_uuid = cards['SIM_UUID']
        self.model_uuid = cards.get('MODEL_UUID', '')
        self.data_file = dmi_util.find_solution_file(
            cards.get('SOLUTION_FILE', ''), cards.get('SOLUTION_FILE_FULL', ''),
            cards.get('SOLUTION_FILE_RELATIVE', ''), main_file
        )
        self.zbud_nam_file_dir = ''

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (tuple): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        return dmi_util.save_to_location_common(self.main_file, new_path, save_type)

    @override
    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (list of str): A list of the main files of the selected components of this type.

        Returns:
            menu_items (list of xmsapi.dmi.MenuItem): A list of menus and menu items to be shown. Note
            that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem. "None" may
            be added to the list to indicate a separator.
        """
        if len(main_file_list) > 1 or not main_file_list:
            return []  # Multi-select or nothing selected

        menu_list = [None]  # None == spacer

        self._add_tree_menu_command('Budget -> Velocity Vectors', '_on_budget_to_velocity_vectors', '', menu_list)
        self._add_tree_menu_command('Budget -> Scalar Datasets', '_on_budget_to_scalar_datasets', '', menu_list)
        self._add_tree_menu_command('Flow Budget...', '_on_flow_budget', '', menu_list)
        self._add_tree_menu_command('Run ZONEBUDGET...', '_run_zonebudget', self.sim_uuid, menu_list)
        menu_list.append(None)
        return menu_list

    def _on_flow_budget(self, query: Query, params: Optional[List[dict]], win_cont: Optional[QWidget]):
        """Runs the flow budget dialog.

        Args:
            query: Object for communicating with GMS
            params: ActionRequest parameters
            win_cont: The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        if not self._cbc_file_exists_or_warn(win_cont):
            return [], []

        sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, self.sim_uuid)
        model_node = self._get_model_node(self.model_uuid, query.project_tree, sim_node)
        flow_budget_runner.run_flow_budget(query, win_cont, sim_node, model_node)
        return [], []

    def _cbc_file_exists_or_warn(self, win_cont: QWidget):
        """Checks for .cbc file and warns user if it doesn't exist.

        Args:
            win_cont: The window container

        Returns:
            (bool): True or False.
        """
        if not os.path.isfile(self.data_file):
            message = f'Could not find budget file at "{self.data_file}". Try re-reading the solution.'
            message_box.message_with_ok(parent=win_cont, message=message)
            return False
        return True

    def _zbud_name_file_exists_or_warn(self, win_cont: QWidget):
        """Check for zone budget files.

        Args:
            win_cont: The window container

        Returns:
            (str): Zone budget name file.
        """
        cbc_dir = os.path.dirname(self.data_file)
        zbud_dir = cbc_dir.rpartition('_output')[0]
        name_file_dir = zbud_dir + '_zone_budget'
        zbud_name_file = os.path.join(name_file_dir, os.path.basename(zbud_dir) + '.nam')
        if not os.path.isfile(zbud_name_file):
            message = (
                f'Could not find zone budget name file at "{zbud_name_file}". '
                'Try re-saving the project and the simulation.'
            )
            message_box.message_with_ok(parent=win_cont, message=message)
            return ''
        return zbud_name_file

    def _zone_budget_exe_or_warn(self, win_cont: QWidget):
        """Check for zone budget executable.

        Args:
            win_cont: The window container

        Returns:
            (str): Zone budget executable file.
        """
        zone_budget_exe = _get_zone_budget_exe_path()
        if not os.path.isfile(zone_budget_exe):
            message = f'Could not find ZONEBUDGET executable file at: \"{zone_budget_exe}\". Failed to run ZONEBUDGET.'
            message_box.message_with_ok(parent=win_cont, message=message)
            return ''
        return zone_budget_exe

    def _get_files_or_warn(self, win_cont: QWidget):
        """Gets the required files or warns the user.

        Args:
            win_cont: The window container

        Returns:
            (tuple): tuple containing:
                - (str): Zone budget executable file.
                - (str): Zone budget name file.
        """
        if not self._cbc_file_exists_or_warn(win_cont):
            return '', ''
        # Cython doesn't like := apparently
        # if (zone_budget_exe := self._zone_budget_exe_or_warn(win_cont)) == '':
        zone_budget_exe = self._zone_budget_exe_or_warn(win_cont)
        if zone_budget_exe == '':
            return '', ''
        # Cython doesn't like := apparently
        # if (zbud_name_file := self._zbud_name_file_exists_or_warn(win_cont)) == '':
        zbud_name_file = self._zbud_name_file_exists_or_warn(win_cont)
        if zbud_name_file == '':
            return '', ''
        return zone_budget_exe, zbud_name_file

    def _run_zonebudget(self, query: Query, params: list[dict], win_cont: QWidget):
        """Opens the "Run Zonebudget" dialog.

        Args:
            query: Object for communicating with GMS
            params: ActionRequest parameters
            win_cont: The window container

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        # Check to see if everything exists first before starting the process feedback dialog or warnings won't show up
        zb_exe, zb_name_file = self._get_files_or_warn(win_cont)
        if not zb_exe:
            return [], []

        comp_dir = os.path.dirname(os.path.dirname(self.main_file))
        zb_in = ZbInput(self.data_file, zb_exe, zb_name_file, self.sim_uuid, self.model_uuid, comp_dir)
        return zone_budget_runner.run(zb_in, query, win_cont)

    def _on_budget_to_scalar_datasets(self, query: Query, params, win_cont: QWidget):
        """Creates datasets from data in the cbc file.

        Args:
            query: Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont: The window container

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        if not self._cbc_file_exists_or_warn(win_cont):
            return [], []

        sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, self.sim_uuid)
        model_node = self._get_model_node(self.model_uuid, query.project_tree, sim_node)
        creator = CbcScalarDatasetCreator(self.data_file, model_node, query)
        messages = creator.create()
        return messages, []

    def _on_budget_to_velocity_vectors(self, query: Query, params, win_cont: QWidget):
        """Creates datasets from data in the cbc file.

        Args:
            query: Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont: The window container

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        if not self._cbc_file_exists_or_warn(win_cont):
            return [], []

        messages = []
        try:
            inputs = VectorInputs()
            inputs.cbc_filepath = self.data_file
            sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, self.sim_uuid)
            inputs.model_node = self._get_model_node(self.model_uuid, query.project_tree, sim_node)
            inputs.grb_filepath = flow_budget_runner.get_binary_grid_file(sim_node.name, inputs.model_node)
            inputs.porosity = self._get_porosity(inputs.grb_filepath, query.copy_project_tree(), query, win_cont)
            if inputs.porosity is None:
                return [], []

            _create_velocity_vectors(inputs, query, win_cont)
        except RuntimeError as e:
            messages.append(('ERROR', str(e)))
            return messages, []
        return messages, []

    def _cell_count_from_grb(self, grb_filepath: Path) -> int:
        """Returns the number of cells in the grid.

        Args:
            grb_filepath (Path): File path of .grb file.

        Returns:
            (int): See description.
        """
        reader = GrbReader(grb_filepath, {'NCELLS', 'NODES'})
        grb_data = reader.read()
        return grb_reader.cell_count_from_grb_data(grb_data)

    def _get_porosity(self, grb_filepath, project_tree: TreeNode, query: Query, win_cont: QWidget) -> float | None:
        """Gets the porosity.

        Args:
            grb_filepath (Path): File path of .grb file.
            project_tree (TreeNode): Project tree.
            query: Object for communicating with GMS
            win_cont: The window container

        Returns:
            See description
        """
        settings = Settings.read_settings(main_file=self.main_file)
        default_choice = settings.get('porosity-choice', 'constant')
        constant = get_from_settings(settings, 'porosity-constant', 0.3)
        mst_uuid = get_from_settings(settings, 'porosity-mst', '')
        dataset_uuid = get_from_settings(settings, 'porosity-dataset', '')
        cell_count = self._cell_count_from_grb(grb_filepath)
        dialog = GetPorosityDialog(default_choice, constant, mst_uuid, dataset_uuid, cell_count, project_tree, win_cont)
        porosity = None
        if dialog.exec() == QDialog.Accepted:
            choice_tuple = dialog.get_choice()
            if choice_tuple[0] == 'constant':
                settings['porosity-constant'] = choice_tuple[1]
                settings['porosity-choice'] = 'constant'
                porosity = choice_tuple[1]
            elif choice_tuple[0] == 'mst':
                settings['porosity-mst'] = choice_tuple[1]
                settings['porosity-choice'] = 'mst'
                porosity = self._porosity_from_mst(project_tree, choice_tuple[1], grb_filepath)
            elif choice_tuple[0] == 'dataset':
                settings['porosity-dataset'] = choice_tuple[1]
                settings['porosity-choice'] = 'dataset'
                porosity = self._porosity_from_dataset(choice_tuple[1], query)
            Settings.write_settings(main_file=self.main_file, settings=settings)
        return porosity

    def _porosity_from_mst(self, project_tree: TreeNode, mst_uuid: str, grb_filepath: Path) -> list[float] | None:
        """Returns the porosity from the MST package.

        Args:
            project_tree (TreeNode): Project tree.
            mst_uuid (str): uuid of the MST package node.

        Returns:
            (float|list[float]): The porosity.
        """
        mst_node = tree_util.find_tree_node_by_uuid(project_tree, mst_uuid)
        if not mst_node:
            return None
        grid_info = GridInfo.from_grb_file(grb_filepath=grb_filepath)
        data = MstData.from_file(mst_node.main_file, 'MST6', grid_info=grid_info)
        porosity_array = data._griddata.get('POROSITY')
        return porosity_array.get_values()

    def _porosity_from_dataset(self, dataset_uuid: str, query) -> list[float] | None:
        """Returns the porosity from the MST package.

        Args:
            dataset_uuid (str): uuid of the dataset node.
            query (xmsapi.dmi.Query): Object for communicating with GMS

        Returns:
            (float|list[float]): The porosity.
        """
        dataset_reader = query.item_with_uuid(dataset_uuid)
        if not dataset_reader:
            return None
        porosity = dataset_reader.values[0]
        return porosity.tolist()

    def _get_model_node(self, model_uuid: str, project_tree: TreeNode, sim_node: TreeNode) -> TreeNode:
        """Returns the model tree node associated with model_uuid, or the first model found if not model_uuid.

        Args:
            model_uuid (str): uuid of the model.
            project_tree (TreeNode): The project tree.
            sim_node (TreeNode): Simulation tree node.

        Returns:
            (TreeNode): See description.
        """
        if model_uuid:
            model_node = tree_util.find_tree_node_by_uuid(project_tree, model_uuid)
        else:
            # We used to not store the model_uuid. Return the first model found
            model_node = tree_util.descendants_of_type(
                sim_node, xms_types=['TI_COMPONENT'], unique_name='GWF6', model_name='MODFLOW 6', only_first=True
            )
            if not model_node:
                model_node = tree_util.descendants_of_type(
                    sim_node, xms_types=['TI_COMPONENT'], unique_name='GWT6', model_name='MODFLOW 6', only_first=True
                )
        return model_node


def _dataset_uuid(query: Query) -> str:
    """Returns the uuid of the solution dataset.

    Args:
        query (Query): Object for communicating with GMS.

    Returns:
        (str): See description.
    """
    comp_uuid = query.current_item_uuid()
    comp_node = tree_util.find_tree_node_by_uuid(query.project_tree, comp_uuid)
    dset_ptr_node = tree_util.descendants_of_type(
        comp_node.parent, xms_types=['TI_DATASET_PTR'], allow_pointers=True, only_first=True
    )
    return dset_ptr_node.uuid if dset_ptr_node else ''


def get_from_settings(settings: dict, key: str, default):
    """Helper to make sure we get a good value, either from settings, or from the default."""
    value = settings.get(key, default)
    if value is None:
        return default
    return value


def get_flow_budget_dialog_data(cbc_node: TreeNode) -> FlowBudgetDialogData:
    """Returns the data needed to run the flow budget dialog.

    Args:
        cbc_node: The CBC component tree node.

    Returns:
        See description.
    """
    dset_ptr_node = tree_util.descendants_of_type(
        cbc_node.parent, xms_types=['TI_DATASET_PTR'], allow_pointers=True, only_first=True
    )
    settings = Settings.read_settings(main_file=cbc_node.main_file)
    setting_precision = 'flow-budget-precision'
    precision = get_from_settings(settings, setting_precision, 2)
    cbc_component = CbcComponent(cbc_node.main_file)
    return FlowBudgetDialogData(dset_ptr_node.uuid, precision, cbc_component.data_file)


def _get_zone_budget_exe_path():
    """Get the path to the Zone Budget program."""
    paths = sim_runner.get_gms_mf6_executable_paths()
    return paths[sim_runner.ExeKey.ZONEBUDGET_6]


def _create_velocity_vectors(inputs: VectorInputs, query: Query, win_cont: QWidget) -> tuple[str, str] | None:
    """Creates a velocity vector and magnitude dataset.

    Args:
        inputs: Everything needed to create the vector datasets.
        query: Object for communicating with GMS.
        win_cont: The window container.

    Returns:
        The error tuple, or None.
    """
    thread = VelocityVectorCreationFeedbackThread(inputs, query)
    process_feedback_dlg.run_feedback_dialog(thread, win_cont)
    return thread.get_error()


class VelocityVectorCreationFeedbackThread(FeedbackThread):
    """Thread for creating the velocity vectors."""
    def __init__(self, inputs: VectorInputs, query: Query):
        """Initializes the class.

        Args:
            inputs: Everything needed to create the vector datasets.
            query: Object for communicating with GMS.
        """
        super().__init__(query, is_export=False)
        self._query = query
        self._inputs = inputs
        self._error = None
        self.display_text |= {
            'title': 'Velocity Vectors',
            'working_prompt': 'Creating velocity vectors...',
            'error_prompt': 'Error(s) encountered.',
            'warning_prompt': 'Warning(s) encountered.',
            'success_prompt': 'Successfully created velocity vectors.',
        }

    def _run(self) -> None:
        """Does the work."""
        self._error = cbc_velocity_vector_dataset_creator.create(self._inputs, self._query)

    def get_error(self) -> tuple[str, str] | None:
        """Returns the error, if any."""
        return self._error
