"""Code to run the package dialog."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass
import os
import shutil
from typing import Any, TYPE_CHECKING

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QApplication, QDialog, QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.components import dmi_util
# I300 TYPE_CHECKING block should have one newline above
if TYPE_CHECKING:  # flake8 is of two minds about the blank line above
    from xms.mf6.components.package_component_base import PackageComponentBase
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.data.model_data_base import ModelDataBase
from xms.mf6.file_io import io_factory
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.gui.dialog_input import DialogInput
from xms.mf6.misc import log_util, util


class Backup:
    """Context manager for making, restoring, and deleting a backup of the components folder."""
    def __init__(self, comp: 'PackageComponentBase', locked: bool):
        """Initializer.

        Args:
            comp: Something derived from PackageComponentBase.
            locked: True if component is locked.
        """
        self._comp = comp
        self._locked = locked
        self._comp_dir = ''
        self._backup_dir = ''

    def __enter__(self):
        """Context manager enter.

        Returns:
            Self.
        """
        if self._comp and not self._locked:
            self._comp_dir = os.path.dirname(self._comp.main_file)
            self._backup_dir = self._comp_dir + "_backup"
            fs.make_or_clear_dir(self._backup_dir)
            shutil.copytree(self._comp_dir, self._backup_dir, dirs_exist_ok=True)
        return self

    def __exit__(self, exc_type, exc_value, exc_tb) -> None:
        """Context manager exit.

        Args:
            exc_type: Exception type.
            exc_value: Exception value.
            exc_tb: Exception traceback.
        """
        if self._comp and os.path.isdir(self._backup_dir):
            shutil.rmtree(self._backup_dir)

    def restore(self) -> None:
        """Restore the backup copy of the component directory."""
        if self._comp:
            fs.clear_folder(self._comp_dir)
            shutil.copytree(self._backup_dir, self._comp_dir, dirs_exist_ok=True)


@dataclass
class RunArgs:
    """Arguments to run_dialog."""
    ftype: str
    dialog_class: Any
    locked: bool
    cell_idxs: list[int]
    ugrid_uuid: str
    help_id: str
    data: BaseFileData


def run_dialog(query: Query, win_cont: QWidget, run_args: RunArgs | None = None, comp=None) -> list[ActionRequest]:
    """Opens the Package dialog.

    Args:
        run_args: Lots of arguments.
        comp: Something derived from PackageComponentBase.
        query: Object for communicating with GMS
        win_cont: The window container.

    Returns:
        list[ActionRequest]: List of action requests.
    """
    locked = query.current_item().locked if not run_args else run_args.locked
    actions = []
    with Backup(comp, locked) as backup:
        try:
            dlg_klass = comp.dialog if not run_args else run_args.dialog_class
            dlg_input = _setup_dlg_input(run_args, comp, query)
            dialog = dlg_klass(dlg_input, win_cont)
            dialog.setModal(True)
            rv = dialog.exec()
            if rv == QDialog.Accepted and not locked:
                write_data(dialog.dlg_input.data)
                actions = dlg_input.actions  # Add actions added by the dialog
                if comp and dlg_input.data.ftype in BaseFileData.displayable_ftypes():
                    comp.ensure_disp_opts_file_exists()
                    # Add action to update the display
                    actions.append(comp.update_displayed_cell_indices(dlg_input.data.model.filename, dlg_input.data))
            elif rv == QDialog.Rejected and dialog.dlg_input.restore_on_cancel is True:
                backup.restore()  # Only need to restore backup if they cancelled after importing or exporting data
        except Exception as error:
            backup.restore()
            logger = log_util.get_logger()
            logger.exception(error)
            raise error
    return actions


def run_dialog_on_file(filepath, base_file_data, locked, win_cont, ftype, dialog_class):
    """Function called when Edit button in List dialog is clicked.

    See ListDialog.on_btn_edit

    Args:
        filepath (str): Filepath of obs file.
        base_file_data (BaseFileData): BaseFileData object.
        locked (bool): True if locked.
        win_cont: The window container.
        ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
        dialog_class: The dialog class. Like SfrDialog.
    """
    reader = io_factory.reader_from_ftype(ftype)
    data = reader.read(
        filepath, mfsim=base_file_data.mfsim, model=base_file_data.model, parent_package_ftype=base_file_data.ftype
    )
    help_id = f'{util.class_fullname(dialog_class)}'
    args = RunArgs(ftype, dialog_class, locked, [], '', help_id, data)
    run_dialog(query=None, win_cont=win_cont, run_args=args)


def _setup_dlg_input(args: RunArgs, comp: 'PackageComponentBase', query: Query) -> DialogInput:
    """Setup and return the DialogInput object.

    Args:
        args: RunArgs.
        query: Object for communicating with GMS

    Returns:
        See description.
    """
    if args:
        return DialogInput(
            data=args.data,
            locked=args.locked,
            selected_cells=args.cell_idxs,
            ugrid_uuid=args.ugrid_uuid,
            query=query,
            help_id=args.help_id,
            mfsim=args.data.mfsim,
        )
    else:
        locked = query.current_item().locked
        mfsim, model, data = comp.read_sim(query)
        ugrid_uuid = _get_ugrid_uuid(model)
        cell_idxs = _get_selected_cells(ugrid_uuid, query)
        help_id = f'{comp.module_name}.open_dialog'
        return DialogInput(
            data=data,
            locked=locked,
            selected_cells=cell_idxs,
            ugrid_uuid=ugrid_uuid,
            query=query,
            help_id=help_id,
            mfsim=mfsim,
        )


def _get_ugrid_uuid(model: ModelDataBase | None) -> str:
    """Return the UGrid uuid if the package is under the model.

    Args:
        model: The model.

    Returns:
        See description.
    """
    if model:
        return dmi_util.ugrid_uuid_from_model_node(model.tree_node)
    return ''


def _get_selected_cells(ugrid_uuid: str, query: Query) -> list[int]:
    """Return the list of selected cells, if any.

    Args:
        ugrid_uuid: uuid of the UGrid.
        query: Object for communicating with GMS

    Returns:
        See description.
    """
    if not ugrid_uuid:
        return []
    selection = query.get_ugrid_selection(ugrid_uuid)
    if selection:
        return selection.get('CELL', [])
    return []


def write_data(data: BaseFileData) -> None:
    """Write the data to disk.

    Args:
        data: The data.
    """
    mfsim_dir = os.path.dirname(data.mfsim.filename)
    writer_options = WriterOptions(
        mfsim_dir=mfsim_dir,
        use_open_close=True,
        use_input_dir=False,
        use_output_dir=False,
        dmi_sim_dir=os.path.join(mfsim_dir, '..'),
        use_periods_db=True
    )
    writer = io_factory.writer_from_ftype(data.ftype, writer_options)
    QApplication.setOverrideCursor(Qt.WaitCursor)
    writer.write(data)
    QApplication.restoreOverrideCursor()
