"""Mf6Importer class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import cProfile
from enum import IntEnum
import io
from pathlib import Path
import pstats
from pstats import SortKey

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import TreeNode
from xms.data_objects.parameters import UGrid as DoGrid
from xms.guipy.dialogs import process_feedback_dlg
from xms.guipy.dialogs.feedback_thread import FeedbackThread

# 4. Local modules
from xms.mf6.components.component_creator import ComponentCreator
from xms.mf6.data.mfsim_data import MfsimData
from xms.mf6.file_io import io_factory, io_util
from xms.mf6.file_io.package_reader_base import PackageReaderBase
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.geom.ugrid_builder import UGridBuilder
from xms.mf6.misc import log_util, util


class ExportEnum(IntEnum):
    """The different export options we support and test."""
    INTERNAL = 0  # Save lists/arrays inside the package files
    EXTERNAL = 1  # Use 'input' and 'output' dirs and OPEN/CLOSE, like when we Save Simulation
    DMI = 2  # Save the files into separate uuid folders under a 'Components' folder


def run_script() -> bool:
    """Imports when called as a script, showing the feedback dialog."""
    query = _create_query()
    thread = ImportFeedbackThread(query)
    return process_feedback_dlg.run_feedback_dialog(thread, None)


def import_sim(mfsim_nam: Path | str, query: Query | None, **kwargs) -> MfsimData:
    """Imports the mfsim.nam simulation.

    Args:
        mfsim_nam: Filepath of mfsim.nam file.
        query: If provided, the components will be created in XMS. Otherwise, the sim is only rewritten.
        **kwargs: Keyword arguments.

    Keyword Args:
        export_enum: ExportEnum = ExportEnum.DMI - How the sim will be exported after reading.
        export_dir: Path | None = None - Where the sim will be exported to after reading.

    Returns:
        The simulation.
    """
    importer = Mf6Importer(mfsim_nam, query, **kwargs)
    return importer.import_sim()


class ImportFeedbackThread(FeedbackThread):
    """Thread for importing the simulation."""
    def __init__(self, query: Query):
        """Initializes the class."""
        super().__init__(query, is_export=True)
        self._query = query
        self._mfsim_nam = query.read_file
        self.display_text |= {
            'title': 'Import MODFLOW 6 simulation',
            'working_prompt': f'Importing MODFLOW 6 simulation \"{self._mfsim_nam}\".',
            'error_prompt': 'Error(s) encountered while importing.',
            'warning_prompt': 'Warning(s) encountered while importing.',
            'success_prompt': f'Successfully imported \"{self._mfsim_nam}\".',
        }

    def _run(self) -> None:
        """Does the work."""
        import_sim(self._mfsim_nam, self._query)


class Mf6Importer:
    """Imports a MODFLOW 6 simulation."""
    def __init__(self, mfsim_nam: Path | str, query: Query | None, **kwargs):
        """Initializes the class.

        Args:
            mfsim_nam: Filepath of mfsim.nam file.
            query: If provided, the components will be created in XMS. Otherwise, the sim is only rewritten.
            **kwargs: Keyword arguments.

        Keyword Args:
            export_enum: ExportEnum = ExportEnum.DMI - How the sim will be exported after reading.
            export_dir: Path | None = None - Where the sim will be exported to after reading.
        """
        _validate_kwargs(**kwargs)
        self._mfsim_nam = mfsim_nam
        self._query = query
        self._export_enum = kwargs.get('export_enum', ExportEnum.DMI)
        self._export_dir = kwargs.get('export_dir', io_util.get_temp_directory())

        self._mfsim: MfsimData | None = None
        self._log = log_util.get_logger()

    def import_sim(self) -> MfsimData:
        """Imports and returns the simulation.

        Returns:
            The simulation.
        """
        if util.file_hack('debug_mf6_profile_import.dbg'):
            self._import_sim_and_profile()
        else:
            self._import_sim()
        return self._mfsim

    def _import_sim(self):
        """Imports the simulation by reading it, rewriting it in our format, and building the components."""
        try:
            self._read()
            self._write()
            if self._query:
                self._create_grids_and_components()
                self._query.send()  # Do this last. Should only be one Query::Send per import script.
            else:
                self._create_grids()
            self._log.info('Simulation import complete.\n')
        except RuntimeError as ex:
            self._log.info(str(ex))
            return

    def _import_sim_and_profile(self):
        """Profiles the import and writes report to C:/temp/profile.txt.

        See https://docs.python.org/3/library/profile.html
        """
        profiler = cProfile.Profile()
        profiler.runcall(self._import_sim)
        s = io.StringIO()
        sortby = SortKey.CUMULATIVE
        ps = pstats.Stats(profiler, stream=s).sort_stats(sortby)
        ps.print_stats()
        with open('C:/temp/profile.txt', 'w') as file:
            file.write(s.getvalue())

    def _read(self):
        """Reads the simulation."""
        self._log.info('Reading simulation...')
        reader = io_factory.reader_from_ftype('MFSIM6')
        self._mfsim = reader.read(filename=self._mfsim_nam)
        if not self._mfsim:
            raise RuntimeError()
        self._read_zone_file()

    def _read_zone_file(self):
        """Looks for .zon files used with ZONEBUDGET and reads them if found."""
        for model in self._mfsim.models:
            gwf_filepath = Path(model.filename)
            for path in gwf_filepath.parent.rglob('*'):  # Recursively walk through files
                # Must have .zon extension and same prefix as GWF name file
                if path.is_file() and path.suffix.lower() == '.zon' and path.stem.lower() == gwf_filepath.stem.lower():
                    reader = io_factory.reader_from_ftype('ZONE6')
                    zone_data = reader.read(
                        str(path), mfsim=self._mfsim, model=self._mfsim.package_from_filename(model.filename)
                    )
                    model.packages.append(zone_data)

    def _write(self):
        """Writes the simulation to self._export_dir."""
        self._log.info('Writing simulation...')
        # PackageReaderBase.importing gets set to False when done reading mfsim.nam, but we're really still importing
        # until we also call self._write(), so set it to True again.
        PackageReaderBase.importing = True
        match self._export_enum:
            case ExportEnum.INTERNAL:
                self._mfsim.filename = str(Path(self._export_dir) / 'mfsim.nam')
                writer_options = WriterOptions(use_open_close=False)
            case ExportEnum.EXTERNAL:
                self._mfsim.filename = str(Path(self._export_dir) / 'mfsim.nam')
                writer_options = WriterOptions(use_open_close=True, use_input_dir=True, use_output_dir=True)
            case ExportEnum.DMI:
                components_dir = Path(self._export_dir) / 'Components'
                components_dir.mkdir(parents=True, exist_ok=True)
                self._mfsim.filename = str(components_dir / 'mfsim.nam')
                writer_options = WriterOptions(use_open_close=True, dmi_sim_dir=str(components_dir))
            case _:
                raise ValueError('Export option not recognized.')
        self._mfsim.write(writer_options)
        PackageReaderBase.importing = False

    def _create_grids(self):
        """Create the grids using the DIS/DISV/DISU package info."""
        self._log.info('Building UGrid...')
        if not self._use_test_grid():
            builder = UGridBuilder()
            builder.build(self._mfsim, None)

    def _create_grids_and_components(self):
        """Creates the grids and the components."""
        self._log.info('Adding items to GMS...')
        creator = ComponentCreator()
        creator.create_components_from_simulation(self._mfsim, self._query, ugrid_uuid='', model_str='')

    def _use_test_grid(self) -> bool:
        """Return True if a test grid is found, which is used instead of building grids from DIS* packages."""
        test_grid_filepath = Path(self._mfsim_nam).with_name('ugrid.xmc')
        if test_grid_filepath.is_file() and self._mfsim.models[0].tree_node:
            dogrid = DoGrid(str(test_grid_filepath))
            self._mfsim.xms_data.set_dogrid(self._mfsim.models[0].tree_node.uuid, dogrid)
            return True
        return False


def _create_query() -> Query:  # pragma no cover - this is always mocked when testing
    """Creates and returns the query, and makes testing easier because this can be mocked."""
    return Query(timeout=300000)


def _set_package_tree_node(data, child: TreeNode):
    """Helper function."""
    child.main_file = data.filename
    data.tree_node = child


def _validate_kwargs(**kwargs) -> None:
    """Raises an exception if any of the kwargs are not recognized.

    Args:
        **kwargs:
    """
    for key in kwargs.keys():
        if key not in {'export_enum', 'export_dir'}:
            raise ValueError(f'Unrecognized keyword argument "{key}"')
