"""Array class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules
import numpy as np
import pandas as pd

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.data.grid_info import GridInfo
from xms.mf6.file_io import io_util
from xms.mf6.misc import util


class ArrayLayer:
    """Array class for use in RCH, EVT, IC, NPF and STO packages."""
    def __init__(
        self,
        storage='CONSTANT',
        numeric_type='float',
        constant=1.0,
        factor=1.0,
        external_filename='',
        binary=False,
        name='',
        shape=None,
        values=None
    ):
        """Initializes the class.

        Args:
            storage (str): 'CONSTANT' or 'ARRAY'.
            numeric_type (str): 'int' or 'float'.
            constant: One value for the entire array_layer (storage = 'Constant').
            factor: Multiplier for the array_layer values (storage = 'Array').
            external_filename (str): Path to the external file where data is stored.
            binary (bool): If true, external file is binary.
            name (str): Name of the array_layer.
            shape: Tuple.
            values: List of values
        """
        self.storage = storage  # 'CONSTANT', 'ARRAY'
        self.numeric_type = numeric_type  # 'int' or 'float'
        self.constant = constant  # One value for entire array_layer (storage = 'Constant')
        self.factor = factor  # Multiplier for the array_layer values (storage = 'Array')
        self.values = values  # list of array_layer values
        self.external_filename = external_filename
        self.binary = binary  # True if binary, external file (filename saved in external_filename)
        self.temp_external_filename = ''  # Used by dialogs when making changes
        self.name = name
        self.shape = shape  # array_layer shape

        if values:
            self.storage = 'ARRAY'

    def __deepcopy__(self, memo):
        """Returns a deep copy of this object.

        Args:
            memo: A memo object.

        Returns:
            See description.
        """
        new_array_layer = ArrayLayer(
            self.storage, self.numeric_type, self.constant, self.factor, '', self.binary, self.name, self.shape
        )
        new_array_layer.values = copy.deepcopy(self.values)
        # Make copies of files
        if self.external_filename:
            new_array_layer.external_filename = io_util.get_temp_filename()
            fs.copyfile(self.external_filename, new_array_layer.external_filename)
        if self.temp_external_filename:
            new_array_layer.temp_external_filename = io_util.get_temp_filename()
            fs.copyfile(self.temp_external_filename, new_array_layer.temp_external_filename)
        return new_array_layer

    def __eq__(self, other):
        """Returns True if self == other."""
        # Don't try to compare external_filename or temp_external_filename. Just compare result of get_values()
        # yapf: disable
        return (
            self.storage == other.storage
            and self.numeric_type == other.numeric_type
            and self.constant == other.constant
            and self.factor == other.factor
            and self.binary == other.binary
            and self.name == other.name
            and self.shape == other.shape
            and self.get_values() == other.get_values()
        )
        # yapf: enable

    def __repr__(self):
        """Returns a string representation of the object to aid in debugging.

        Returns:
            (str): A string representation of the class object.
        """
        return (
            f'storage: {self.storage}, '
            f'external_filename: {self.external_filename}, '
            f'temp_external_filename: {self.temp_external_filename}, '
            f'factor: {self.factor}, '
            f'numeric_type: {self.numeric_type}, '
            f'name: {self.name}'
        )

    def get_values(self, apply_factor: bool = False):
        """Returns the values.

        If the array_layer is set to use a constant value, a full list, size of shape, will be returned.

        Args:
            apply_factor: If True, the values are multiplied by the factor.

        Returns:
            See description.
        """
        if self.shape is None:
            return []

        caster = io_util.type_caster_from_string(self.numeric_type)
        factor = self.factor if apply_factor else caster(1.0)

        count = (self.shape[0] * self.shape[1])
        if self.storage == 'CONSTANT':
            return [self.constant * factor] * count
        elif self.values:
            if isinstance(self.values[0], str):
                self.values = [caster(word) * factor for word in self.values]
            return self.values
        else:
            filename = ''
            if self.temp_external_filename:  # Look at temp first
                filename = self.temp_external_filename
            elif self.external_filename:
                filename = self.external_filename
            with open(filename, 'r') as file:
                values, _extra_words_after_array = io_util.read_array_values(file, count)
                caster = io_util.type_caster_from_string(self.numeric_type)
                # Can't do int('2.0') so always go to float first
                values = [caster(float(word)) * factor for word in values]
                return values

    def set_values(self, values, shape, use_constant=True):
        """Sets the values and shape, and sets storage to 'CONSTANT' or 'ARRAY'.

        The values will be in RAM until you decide to write them. External filenames are cleared.

        Args:
            values (list(float or int): The values.
            shape: Shape of the array_layer.
            use_constant (bool): If true, checks if all values are equal and if so, uses a constant instead.
        """
        if use_constant and util.check_all_equal(values):
            self.storage = 'CONSTANT'
            self.constant = values[0]
        else:
            self.storage = 'ARRAY'
            self.values = values
        self.shape = shape
        self.external_filename = ''
        self.temp_external_filename = ''

    def set_constant(self, constant: int | float) -> None:
        """Sets the array_layer to be a constant value.

        Args:
            constant (int|float): The constant value
        """
        self.storage = 'CONSTANT'
        self.numeric_type = 'float' if isinstance(constant, float) else 'int'
        self.constant = constant
        self.factor = 1.0
        self.values = []
        self.external_filename = ''
        self.temp_external_filename = ''

    def combine_values(self, new_values, shape):
        """Combines new values with existing values, overwriting existing where new is not util.XM_NODATA.

        The values will be in RAM until you decide to write them. External filenames are cleared.

        Args:
            new_values (list(float or int): The new values.
            shape: Shape of the array_layer.
        """
        old_values = self.get_values()
        if len(old_values) != len(new_values):
            raise RuntimeError('Array.combine_values: sizes are not equal.')

        for i in range(len(new_values)):
            if new_values[i] != util.XM_NODATA:
                old_values[i] = new_values[i]
        self.set_values(old_values, shape)

    @staticmethod
    def number_of_values_and_shape(layered: bool, grid_info: GridInfo):
        """Returns tuple of nvals and shape.

        Args:
            layered: True if layered.
            grid_info: Number of rows, cols etc.

        Returns:
            (tuple): tuple containing:
                - nvals (int): Number of values.
                - shape (tuple): tuple containing:
                    - shape0 (int): First dimension.
                    - shape1 (int): Second dimension.
        """
        if not grid_info:
            return 0, (0, 0)

        shape1 = 1
        if layered:
            if grid_info.nodes != -1:  # 'LAYERED' keyword can actually be used with DISU. GNNNH!!!
                shape0 = nvals = grid_info.nodes
            elif grid_info.ncpl != -1:
                shape0 = nvals = grid_info.ncpl
            else:
                nvals = grid_info.nrow * grid_info.ncol
                shape0 = grid_info.nrow
                shape1 = grid_info.ncol
        else:
            if grid_info.nodes != -1:
                shape0 = nvals = grid_info.nodes
            elif grid_info.ncpl != -1:
                shape0 = nvals = grid_info.ncpl * grid_info.nlay
            else:
                shape0 = nvals = grid_info.nrow * grid_info.ncol * grid_info.nlay
        shape = (shape0, shape1)
        return nvals, shape

    def to_dataframe(self) -> pd.DataFrame:
        """Returns the data as a pd.DataFrame.

        Returns:
            See description.
        """
        narray = np.array(self.get_values()).reshape(self.shape)
        df = pd.DataFrame(data=narray)
        return df
