"""AtsData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.gui import gui_util, units_util


class AtsData(BaseFileData):
    """Class to hold the ATS package data."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'ATS6'
        self.list_blocks = {'PERIODDATA': ''}  # (list blocks -> filename)

    # @overrides
    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        The columns depend on the DIS package in use and the AUX variables.
        The package specific and AUX columns are type object because they
        might contain time series strings.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        return self.package_column_info()

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info('')
        time_units = units_util.string_from_units(self, units_util.UNITS_TIME)
        return {
            names.index('IPERATS'): 'Period number to designate for adaptive time stepping.',
            names.index('DT0'):
                f'Initial time step length for period iperats. If dt0 is zero, then the final'
                f' step from the previous stress period will be used as the initial time step.'
                f' {time_units}',
            names.index('DTMIN'):
                f'Minimum time step length for this period. This value must be greater than'
                f' zero and less than dtmax. dtmin must be a small value in order to ensure'
                f' that simulation times end at the end of stress periods and the end of the'
                f' simulation. A small value, such as 1.e-5, is recommended. {time_units}',
            names.index('DTMAX'):
                f'Maximum time step length for this period. This value must be greater than'
                f' dtmin. {time_units}',
            names.index('DTADJ'): 'Time step multiplier factor for this period.',
            names.index('DTFAILADJ'): 'Divisor of the time step length when a time step fails to converge.',
        }

    # @overrides
    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        You should override this method.

        Args:
            block (str): Name of the block.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        columns = {
            'IPERATS': (np.int32, 1),
            'DT0': (np.float64, 0.0),
            'DTMIN': (np.float64, 1.0e-5),
            'DTMAX': (np.float64, 1.0),
            'DTADJ': (np.float64, 2.0),
            'DTFAILADJ': (np.float64, 5.0),
        }

        return gui_util.column_info_tuple_from_dict(columns)

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Adaptive Time Step Utility (ATS)'
