"""Block class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.array import Array


class Block:
    """Class that holds block data, like GRIDDATA or CONNECTIONDATA."""
    def __init__(self, name: str, array_names: list[str]):
        """Initializer.

        Args:
            name: Name for this block (e.g. GRIDDATA)
            array_names: List of the array names this block can have
        """
        self._name = name
        self._array_names: list[str] = array_names
        self._arrays: dict[str, Array] = {}

    def __deepcopy__(self, memo):
        """Returns a deep copy of this object.

        Args:
            memo: A memo object.

        Returns:
            See description.
        """
        new_block = Block(self.name, self.names)
        new_block._arrays = copy.deepcopy(self._arrays)
        return new_block

    def __eq__(self, other):
        """Returns True if self == other."""
        # yapf: disable
        return (
            self.name == other.name
            and self.names == other.names
            and self.arrays == other.arrays
        )
        # yapf: enable

    @property
    def name(self) -> str:
        """Return the name of this block.

        Returns:
            See description.
        """
        return self._name

    @property
    def names(self) -> list[str]:
        """Return a list of all the array names that this block MIGHT include.

        Returns:
            See description.
        """
        return self._array_names

    @property
    def existing_names(self) -> list[str]:
        """Return a list of names of the arrays that actually exist.

        Returns:
            See description.
        """
        return [array.array_name for array in self._arrays.values()]

    @property
    def arrays(self) -> list[Array]:
        """Return a list of all the arrays in this block.

        Returns:
            See description.
        """
        return list(self._arrays.values())

    def array(self, name: str) -> Array | None:
        """Return the array with the given name.

        Args:
            name: Name of the array.

        Returns:
            See description.
        """
        return self._arrays.get(name)

    def add_array(self, array: Array) -> Array | None:
        """Add the array to this block.

        Args:
            array: The array.
        """
        self._arrays[array.array_name] = array
        return array

    def delete_array(self, name: str) -> None:
        """Delete the array with the given name.

        Args:
            name: Name of the array.
        """
        if name in self._arrays:
            del self._arrays[name]

    def delete_arrays(self) -> None:
        """Delete all the arrays (but keeps the list of array names)."""
        self._arrays.clear()

    def has(self, name: str) -> bool:
        """Return True if this block has an array with the given name.

        Args:
            name: Name of the array.

        Returns:
            See description.
        """
        return name in self._arrays

    def dump_to_temp_files(self, temp_file_list: list[str] | None = None):
        """Writes array values in RAM to external files.

        Used when importing and in map from coverage.

        Args:
            temp_file_list: list of temporary file names
        """
        for array in self._arrays.values():
            array.dump_to_temp_files(temp_file_list)
