"""GwxGwxData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.exchanges.exchange_data_base import ExchangeDataBase
from xms.mf6.gui import gui_util


class GwxGwxDataBase(ExchangeDataBase):
    """Data class to hold the info from a GWF-GWF or GWT-GWT exchange package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.list_blocks = {'EXCHANGEDATA': ''}  # (list blocks -> filename)

    # @overrides
    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        The columns depend on the DIS package in use and the AUX variables.
        The package specific and AUX columns are type object because they
        might contain time series strings.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        if block.upper() == 'EXCHANGEDATA':
            id_columns = data_util.get_id_column_dict(self.grid_info())
            cellidm1_columns = dict((f'{key}M1', value) for (key, value) in id_columns.items())
            cellidm2_columns = dict((f'{key}M2', value) for (key, value) in id_columns.items())
            columns = {
                **cellidm1_columns,
                **cellidm2_columns,
                **{
                    'IHC': (np.int32, 1),
                    'CL1': (np.float64, 0.0),
                    'CL2': (np.float64, 0.0),
                    'HWVA': (np.float64, 0.0),
                }
            }
            self.add_aux_columns_to_dict(columns, use_aux=use_aux)
        else:
            raise AssertionError()  # Error!

        names, types, defaults = gui_util.column_info_tuple_from_dict(columns)
        return names, types, defaults

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info(block)
        if block.upper() == 'EXCHANGEDATA':
            return {
                names.index('IHC'): 'Flag indicating the direction between node n and all of its m connections',
                names.index('CL1'): 'Distance between the center of cell 1 and the its shared face with cell 2',
                names.index('CL2'): 'Distance between the center of cell 2 and the its shared face with cell 1',
                names.index('HWVA'):
                    'Horizontal width of the flow connection between cell 1 and cell 2 if IHC > 0,'
                    ' or it is the area perpendicular to flow of the vertical connection between'
                    ' cell 1 and cell 2 if IHC = 0',
            }
        return {}

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        You should override this method.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        # TODO:
        return [], {}, {}

    def dialog_title(self):
        """Returns the title to show in the dialog.

        You should override this method.

        Returns:
            (str): The dialog title.
        """
        return ''

    def block_with_aux(self):
        """Returns the name of the block that can have aux variables.

        Returns:
            (str): The name of the block that can have aux variables.
        """
        return 'EXCHANGEDATA'

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return None
