"""EstData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import units_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxField


class EstData(GriddataBase):
    """Data class to hold the info from an EST package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
               it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'EST6'
        arrays = ['POROSITY', 'DECAY_WATER', 'DECAY_SOLID', 'HEAT_CAPACITY_SOLID', 'DENSITY_SOLID']
        self.add_block('GRIDDATA', arrays)

    def get_units(self, array_name: str) -> str:
        """Returns the units string for the array.

        Args:
            array_name (str): The name of an array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        return units_util.UNITS_UNITLESS

    def is_int_array(self, array_name):
        """Returns True if the array is integers.

        Args:
            array_name (str): The name of an array.

        Returns:
            (bool): True or False
        """
        return False

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name (str): The name of an array.

        Returns:
            (bool): True or False
        """
        return array_name in ['POROSITY', 'HEAT_CAPACITY_SOLID', 'DENSITY_SOLID']

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Energy Storage and Transfer (EST) Package'

    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                Checkbox(
                    'ZERO_ORDER_DECAY_WATER',
                    brief='Use zero order decay in the aqueous phase (define DECAY_WATER in the GRIDDATA block)'
                ),
                Checkbox(
                    'ZERO_ORDER_DECAY_SOLID',
                    brief='Use zero order decay in the solid phase (define DECAY_SOLID in the GRIDDATA block)'
                ),
                CheckboxField('DENSITY_WATER', brief='Density of water', type_='float', value=1000.0),
                CheckboxField('HEAT_CAPACITY_WATER', brief='Heat capacity of water', type_='float', value=4184.0),
                CheckboxField(
                    'LATENT_HEAT_VAPORIZATION', brief='Latent heat of vaporization', type_='float', value=2453500.0
                ),
            ]
        )

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'POROSITY':
            txt = (
                'Mobile domain porosity, defined as the mobile domain pore volume per mobile domain volume. The GWE'
                ' model does not support the concept of an immobile domain in the context of heat transport.'
            )
        elif tab_text == 'DECAY_WATER':
            txt = (
                'Rate coefficient for zero-order decay for the aqueous phase of the mobile domain. A negative value'
                ' indicates heat (energy) production. The dimensions of zero-order decay in the aqueous phase are'
                ' energy per length cubed (volume of water) per time. Zero-order decay in the aqueous phase will have'
                ' no effect on simulation results unless ZERO_ORDER_DECAY_WATER is specified in the options block.'
            )
        elif tab_text == 'DECAY_SOLID':
            txt = (
                'Rate coefficient for zero-order decay for the solid phase. A negative value indicates heat (energy)'
                ' production. The dimensions of zero-order decay in the solid phase are energy per mass of solid per'
                ' time. Zero-order decay in the solid phase will have no effect on simulation results unless'
                ' ZERO_ORDER_DECAY_SOLID is specified in the options block.'
            )
        elif tab_text == 'HEAT_CAPACITY_SOLID':
            txt = (
                'Mass-based heat capacity of dry solids (aquifer material). For example, units of J/kg/C may be used'
                ' (or equivalent).'
            )
        elif tab_text == 'DENSITY_SOLID':
            txt = (
                'User-specified value of the density of aquifer material not considering the voids. Value will remain'
                ' fixed for the entire simulation. For example, if working in SI units, values may be entered as'
                ' kilograms per cubic meter.'
            )
        return txt
