"""EvtArrayData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.gwf.array_package_data import ArrayPackageData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import units_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxButton, CheckboxField


class EvtArrayData(ArrayPackageData):
    """Data class to hold the info from a EVT array-based package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'EVT6'
        self._griddata_names = ['IEVT', 'SURFACE', 'RATE', 'DEPTH']
        self.readasarrays = True
        self._layer_indicator = 'IEVT'

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name (str): The name of an array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() in ['SURFACE', 'RATE', 'DEPTH']

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Evapotranspiration (EVT) Package'

    @override
    def get_units(self, array_name: str) -> str:
        """Returns a units string.

        Args:
            array_name (str): The name of a array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        units_str = ''
        match array_name:
            case 'SURFACE':
                units_str = units_util.UNITS_LENGTH
            case 'RATE':
                units_str = units_util.UNITS_INFILTRATION
            case 'DEPTH':
                units_str = units_util.UNITS_LENGTH
            case _:
                return ''  # This is an error
        return units_str

    def is_layer_indicator(self, array_name):
        """Returns True if the array is a layer indicator ('IEVT', 'IRCH').

        Returns:
            (bool): True or False
        """
        return array_name.upper() == 'IEVT'

    def array_supports_time_array_series(self, array_name):
        """Returns True if the array can have time-array series.

        Returns:
            (bool): True or False
        """
        if array_name != 'IEVT' and array_name != 'SURFACE' and array_name != 'DEPTH':
            return True
        return False

    def map_info(self, feature_type):
        """Returns info needed for Map from Coverage.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
        """
        if feature_type != 'polygons':
            return {}
        return {'Name': None, 'SURFACE': None, 'RATE': None, 'DEPTH': None}

    def map_import_info(self, feature_type):
        """Returns info needed for mapping shapefile or transient data file to package data.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): See description
        """
        return {}

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'IEVT':
            txt = 'Layer number that defines the layer in each vertical column where evapotranspiration is applied.'
        elif tab_text == 'SURFACE':
            txt = 'Elevation of the ET surface.'
        elif tab_text == 'RATE':
            txt = 'Maximum ET flux rate.'
        elif tab_text == 'DEPTH':
            txt = 'ET extinction depth.'
        return txt

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                Checkbox('READASARRAYS', brief='Array-based input will be used'),
                Checkbox('FIXED_CELL', brief='Do not reassign to underlying cell if cell is inactive'),
                CheckboxButton(
                    'AUXILIARY',
                    button_text='Auxiliary Variables...',
                    check_box_method='on_chk_auxiliary',
                    button_method='on_btn_auxiliary'
                ),
                CheckboxField(
                    'AUXMULTNAME', brief='Auxiliary variable used as multiplier of well flow rate', type_='str'
                ),
                Checkbox('PRINT_INPUT', brief='Print input to listing file'),
                Checkbox('PRINT_FLOWS', brief='Print flows to listing file'),
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                CheckboxButton(
                    'TAS6 FILEIN',
                    brief='Time-array-series files',
                    button_text='Files...',
                    button_method='on_btn_tas6_filein'
                ),
                CheckboxButton(
                    'OBS6 FILEIN',
                    brief='Observation files',
                    button_text='Files...',
                    button_method='on_btn_obs6_filein'
                ),
            ]
        )
